(ns tidy.core
  (:require [clojure.core.async :as core-async]
            [clojure.main :refer [repl-read]]
            [utilis.fn :refer [fsafe]]
            [utilis.map :refer [compact map-vals map-keys]]
            [taoensso.timbre :as log]
            [clojure.set :refer [difference]]
            [clojure.string :as st])
  (:import [java.util.concurrent LinkedBlockingQueue]))

(defn current-thread-name [] (:name (bean (Thread/currentThread))))

(defn nano-offset
  "Debug tool for timing order of executions"
  []
  (float (/ (mod (System/nanoTime) 1000000000) 1000000000)))

(defn log-info
  [& args]
  (log/info (nano-offset) args))

;;; Declarations

(declare subscribe-reaction* unsubscribe-reaction*
         subscribe-ratom* unsubscribe-ratom*
         dispose* do-work! schedule-work! deref*
         reaction?)

(def ^:dynamic *deref-context* nil)
(def ^:private work-complete-token :tidy/complete-token)
(def ^:private control-token :tidy/control-token)
(def ^:private default-reaction-buffer-size 1)
(def ^:private default-worker-buffer-size 1024)

(defprotocol IReaction
  (dispose [this])
  (disposed? [this])
  (started? [this]))

(defprotocol ISubscribable
  (subscribe [this key opts])
  (unsubscribe [this key]))

(defprotocol IIdentifiable
  (id [this]))

(defprotocol IReactiveAtom)

(defprotocol IInitializable
  (initialized? [this]))

;;; Worker Pool

(defonce worker-pools (ref nil))

(defn start-worker-pool
  ([] (start-worker-pool (.availableProcessors (Runtime/getRuntime))))
  ([n] (start-worker-pool n default-worker-buffer-size))
  ([n i]
   (let [queue (LinkedBlockingQueue. i)]
     (dotimes [_ n]
       (future
         (loop []
           (let [work (.take queue)]
             (when-not (= work work-complete-token)
               (try (do-work! work)
                    (catch Exception e
                      (println e "Exception occurred doing work")))
               (recur))))
         (log/info "Exited worker" n)))
     {:queue queue :n n})))

(defn shutdown-worker-pool
  [worker-pool]
  (dotimes [_ (:n worker-pool)]
    (.offer (:queue worker-pool) work-complete-token)))

;;; Records

(defrecord RAtom [id* backing-atom report-queue control-queue]
  IReactiveAtom

  IIdentifiable
  (id [this] id*)

  IInitializable
  (initialized? [this]
    (not= :tidy/none @backing-atom))

  clojure.lang.IDeref
  (deref [this] (deref* this))

  clojure.lang.IAtom
  (reset [this new-value] (.reset ^clojure.lang.IAtom backing-atom new-value))
  (swap [this f]          (.swap ^clojure.lang.IAtom backing-atom f))
  (swap [this f x]        (.swap ^clojure.lang.IAtom backing-atom f x))
  (swap [this f x y]      (.swap ^clojure.lang.IAtom backing-atom f x y))
  (swap [this f x y more] (.swap ^clojure.lang.IAtom backing-atom x y more))

  clojure.lang.IRef
  (addWatch [this key f]  (.addWatch ^clojure.lang.IAtom backing-atom key f) this)
  (removeWatch [this key] (.removeWatch ^clojure.lang.IAtom backing-atom key) this)
  (getWatches [this]      (.getWatches backing-atom))

  ISubscribable
  (subscribe [this key opts] (subscribe-ratom* this key opts))
  (unsubscribe [this key]    (unsubscribe-ratom* this key))

  Object
  (toString [this] (str "RAtom[id=" id* ",value=" @this "]"))

  )

(defn ratom?
  [x]
  (satisfies? IReactiveAtom x))

(defn ratom
  ([] (ratom :tidy/none))
  ([initial-state] (ratom (str "ratom_" (gensym)) initial-state))
  ([id initial-state]
   (RAtom. id (atom initial-state)
           (LinkedBlockingQueue. 1)
           (doto (LinkedBlockingQueue. 1)
             (.put control-token)))))

(defrecord Reaction [id* f backing-atom listeners state
                     job-queue report-queue control-queue
                     watching opts]
  IReaction
  (dispose [this] (dispose* this))
  (disposed? [this]
    (boolean
     (#{:tidy/marked-for-disposal
        :tidy/disposed} @state)))
  (started? [this]
    (and (= :tidy/started @state)
         (->> @watching
              (filter reaction?)
              (every? started?))))

  IInitializable
  (initialized? [this]
    (boolean
     (and
      (or (empty? @watching)
          (and (:best-effort? opts)
               (some initialized? @watching))
          (and (not (:best-effort? opts))
               (every? initialized? @watching)))
      (started? this))))

  ISubscribable
  (subscribe [this key opts] (subscribe-reaction* this key opts))
  (unsubscribe [this key]    (unsubscribe-reaction* this key))

  IIdentifiable
  (id [this] id*)

  clojure.lang.IDeref
  (deref [this] (deref* this))

  Object
  (toString [this] (str "Reaction[id=" id* ",value=" @this ",state=" @state "]")))

(defn make-reaction
  ([f] (make-reaction (str "reaction_" (gensym)) f))
  ([id f] (make-reaction id f nil))
  ([id f opts]
   (Reaction.
    id f
    (atom :tidy/none)
    (atom nil)
    (atom :tidy/idle)
    (LinkedBlockingQueue.
     (or (:buffer-size opts)
         default-reaction-buffer-size))
    (LinkedBlockingQueue. 1)
    (doto (LinkedBlockingQueue. 1)
      (.put control-token))
    (atom #{})
    (merge {:best-effort? false} opts))))

(defmacro reaction
  [& body]
  `(make-reaction
    (fn [] ~@body)))

(defn reaction?
  [x]
  (satisfies? IReaction x))

(defn on-state
  [rct state callback]
  (let [watch-id (str :tidy/await-state "-" (gensym))
        called? (ref false)
        maybe-signal! #(when (= % state)
                         (when (dosync
                                (boolean
                                 (when-not @called?
                                   (ref-set called? true))))
                           (remove-watch (:state rct) watch-id)
                           (callback)))]
    (add-watch
     (:state rct) watch-id
     (fn [_ _ _ state*]
       (maybe-signal! state*)))
    (maybe-signal! @(:state rct))))

(defn await-state
  [rct state]
  {:pre [(reaction? rct)
         (#{:tidy/subscribed
            :tidy/started
            :tidy/marked-for-disposal
            :tidy/disposed} state)]}
  (let [queue (LinkedBlockingQueue. 1)]
    (on-state rct state #(.offer queue :tidy/complete-token))
    (.take queue)
    @(:state rct)))

(defn await-disposed
  [rct]
  (await-state rct :tidy/disposed))

(defn await-subscribed
  [rct]
  (await-state rct :tidy/subscribed))

(defn await-started
  [rct]
  (await-state rct :tidy/started))

(defn await-marked-for-disposal
  [rct]
  (await-state rct :tidy/marked-for-disposal))

;;; Record Print Helpers

(defmethod clojure.pprint/simple-dispatch Reaction [o]
  ((get-method clojure.pprint/simple-dispatch clojure.lang.IRecord) o))

(defmethod clojure.pprint/simple-dispatch RAtom [o]
  ((get-method clojure.pprint/simple-dispatch clojure.lang.IRecord) o))

(prefer-method print-method clojure.lang.IDeref clojure.lang.IPersistentMap)
(prefer-method print-method clojure.lang.IDeref clojure.lang.IRecord)
(prefer-method print-method clojure.lang.IDeref java.util.Map)

;;; Private

(defn- ensure-worker-pool!
  [pool-name pool-size worker-buffer-size]
  (when-not (get @worker-pools pool-name)
    (let [pool (start-worker-pool pool-size worker-buffer-size)]
      (when-not (= pool (get (dosync (alter worker-pools assoc pool-name pool)) pool-name))
        (shutdown-worker-pool pool)))))

(defn- ensure-worker-pools!
  []
  (ensure-worker-pool! :compute 1 1024)
  (ensure-worker-pool! :report 1 1024)
  (ensure-worker-pool! :schedule 1 default-worker-buffer-size))

(defn- schedule-work!
  ([reaction job-type] (schedule-work! reaction job-type nil))
  ([reaction job-type job-params]
   (ensure-worker-pools!)
   (when (or (not (reaction? reaction))
             (not (disposed? reaction))
             (when-let [state (:state reaction)]
               (and (= @state :tidy/marked-for-disposal)
                    (= job-type :tidy/dispose))))
     (let [job {:reaction reaction
                :job {:type job-type
                      :params job-params
                      :id (str "job_" (gensym))}}]
       (when-not (.offer
                  (-> @worker-pools :schedule :queue)
                  {:type :tidy/schedule :params job})
         (throw (ex-info "Unable to schedule work. Queue is full." {:job job})))))))

(defn- schedule-run!
  ([reaction] (schedule-run! reaction nil))
  ([reaction params]
   (schedule-work! reaction :tidy/run params)))

(defn- schedule-notify-subscribers!
  [reaction params]
  (schedule-work! reaction :tidy/notify-subscribers params))

(defn- subscribe-reaction*
  [this key opts]
  (schedule-work! this :tidy/subscribe {:key key :opts opts}))

(defn- unsubscribe-reaction*
  [this key]
  (schedule-work! this :tidy/unsubscribe {:key key}))

(defn- subscribe-ratom*
  [this key opts]
  (when-let [on-value (:on-value opts)]
    (add-watch this key (fn [_ _ _ value] (on-value value)))))

(defn- unsubscribe-ratom*
  [this key]
  (remove-watch this key))

(defn- dispose*
  [this]
  (schedule-work! this :tidy/dispose))

(defn- on-value-listeners
  [this]
  (->> @(:listeners this)
       (filter (fn [[k v]] (:on-value v)))
       (into {})))

(defn- on-unsubscribe-listeners
  [this]
  (->> @(:listeners this)
       (filter (fn [[k v]] (:on-unsubscribe v)))
       (into {})))

(defn- deref*
  [this]
  (let [value
        (if (or *deref-context* (not (reaction? this)))
          (let [value (let [values (:values *deref-context*)]
                        (if (and (seq values)
                                 (contains? values this))
                          (get values this)
                          @(:backing-atom this)))]
            (when-let [captured (:captured *deref-context*)]
              (swap! captured conj this))
            value)
          ((:f this)))]
    (when-not (= value :tidy/none) value)))

(defn- do-schedule!
  [work]
  (let [{:keys [type params]} work
        job params
        {:keys [reaction]} job
        job-type (-> job :job :type)]
    (.take (:control-queue reaction))
    (try
      (if (= job-type :tidy/notify-subscribers)
        (do (.put (:report-queue reaction) job)
            (.put (-> @worker-pools :report :queue) (:report-queue reaction)))
        (do (.put (:job-queue reaction) job)
            (.put (-> @worker-pools :compute :queue) (:job-queue reaction))))
      (catch Exception e
        (log/error e "Exception occurred scheduling work")
        (.offer (:control-queue reaction) control-token)))))

(defn- do-run!
  [{:keys [type job reaction]}]
  (when (or (not (-> job :params :allowed-states))
            ((-> job :params :allowed-states) @(:state reaction)))
    (let [captured (atom #{})
          f (:f reaction)]
      (binding [*deref-context*
                {:captured captured
                 :values (-> job :params :values)}]
        (let [pre-started? (started? reaction)
              result (f)
              watching-old @(:watching reaction)
              watching-new @captured]
          (reset! (:watching reaction) watching-new)
          (doseq [c (difference watching-new watching-old)]
            (when (and (not pre-started?) (reaction? c))
              (on-state
               c :tidy/started
               (fn []
                 (when (and (get @(:watching reaction) c)
                            (not (started? reaction))
                            (not (disposed? reaction)))
                   (schedule-run! reaction)))))
            (subscribe
             c reaction
             {:on-value (fn [value]
                          (when-not (#{:tidy/disposed
                                       :tidy/marked-for-disposal} @(:state reaction))
                            (await-started reaction)
                            (schedule-run! reaction {:values {c value}})))
              :on-unsubscribe (fn [] (swap! (:watching reaction) disj c))
              :on-dispose (fn [] (dispose reaction))}))
          (doseq [c (difference watching-old watching-new)]
            (unsubscribe c reaction))
          (reset! (:backing-atom reaction) result)
          (when (and (not pre-started?)
                     (->> watching-new
                          (filter reaction?)
                          (every? started?)))
            (reset! (:state reaction) :tidy/started))
          (when (and pre-started? (initialized? reaction))
            (schedule-notify-subscribers!
             reaction
             {:result result
              :subscribers (on-value-listeners reaction)})))))))

(defn- do-notify-subscribers!
  [{:keys [type job reaction]}]
  (let [{:keys [result subscribers]} (:params job)]
    (doseq [[_ {:keys [on-value]}] subscribers]
      (when (fn? on-value)
        (on-value result)))))

(defn- do-subscribe!
  [{:keys [type job reaction]}]
  (let [{:keys [key opts]} (:params job)]
    (when (get @(:listeners reaction) key)
      (println (str "WARN: overwriting listener key " (or key "nil") " for reaction " (str reaction))))
    (swap! (:listeners reaction) assoc key (dissoc opts :on-dispose))
    (when-let [on-dispose (:on-dispose opts)]
      (on-state reaction :tidy/disposed on-dispose))
    (let [pre-state @(:state reaction)]
      (when (= :tidy/idle pre-state)
        (reset! (:state reaction) :tidy/subscribed)
        (schedule-run! reaction {:tidy/allowed-states #{:tidy/subscribed}})))))

(defn- do-unsubscribe!
  [{:keys [type job reaction]}]
  (let [{:keys [key]} (:params job)]
    (let [had-on-value-listeners? (boolean (seq (on-value-listeners reaction)))]
      (let [unsubscribe-listeners (on-unsubscribe-listeners reaction)]
        (doseq [[_ {:keys [on-unsubscribe]}] unsubscribe-listeners] (on-unsubscribe))
        (swap! (:listeners reaction) dissoc key)
        (when (and had-on-value-listeners? (empty? (on-value-listeners reaction)))
          (reset! (:state reaction) :tidy/marked-for-disposal)
          (dispose reaction))))))

(defn- do-dispose!
  [{:keys [type job reaction]}]
  (when (not= :tidy/disposed @(:state reaction))
    (doseq [w @(:watching reaction)]
      (unsubscribe w reaction))
    (if-let [listeners (not-empty (on-value-listeners reaction))]
      (doseq [[k _] listeners]
        (unsubscribe reaction k))
      (reset! (:state reaction) :tidy/disposed))))

(defn- do-work!
  [work]
  (if (and (map? work) (= :tidy/schedule (:type work)))
    (do-schedule! work)
    (let [work (.take work)
          {:keys [type job reaction]} work]
      (try
        (case (:type job)
          :tidy/run (do-run! work)
          :tidy/notify-subscribers (do-notify-subscribers! work)
          :tidy/subscribe (do-subscribe! work)
          :tidy/unsubscribe (do-unsubscribe! work)
          :tidy/dispose (do-dispose! work))
        (catch Exception e
          (println e "Exception occurred doing work"))
        (finally (.offer (:control-queue reaction) control-token))))))
