(ns tidy.subs
  (:require [tidy.core :as tidy]
            [clojure.core.async :as core-async]
            [utilis.fn :refer [fsafe]]
            [utilis.map :refer [compact map-vals]]
            [taoensso.timbre :as log])
  (:import [java.util.concurrent LinkedBlockingQueue]))

(declare keyword->string queue->seq)

(defonce query->reaction (atom {}))
(defonce handlers (atom {}))
(def monitor (Object.)) ;; locking for now just to get it working

(defn clear-cache-for-id!
  [id]
  (locking monitor
    (swap! query->reaction
           (comp (partial into {})
              (partial remove (fn [[[id* & _] _]] (= id* id)))))))

(defn reg-sub-raw
  [id f]
  (clear-cache-for-id! id)
  (locking monitor
    (swap!
     handlers assoc id
     {:f (fn [ctx query]
           (let [r (f ctx query)]
             (when (not (tidy/reaction? r))
               (throw
                (ex-info "Result of computation parameter in 'reg-sub-raw' must be a reaction"
                         {:id id
                          :r r})))
             r))})))

(defn subscribe
  [query]
  (locking monitor
    (let [rct (or (get @query->reaction query)
                  (if-let [handler-fn (:f (get @handlers (first query)))]
                    (let [reaction (handler-fn nil query)
                          listener-id (str "tidy/subs.subscribe." query)]
                      (tidy/subscribe
                       reaction listener-id
                       {:on-dispose
                        (fn []
                          (locking monitor
                            (when (= reaction (get @query->reaction query))
                              (swap! query->reaction dissoc query))))})
                      (swap! query->reaction assoc query reaction)
                      reaction)
                    (throw (ex-info "No handler registered for query" {:query query}))))]
      (when (and (tidy/disposed? rct) (not tidy/*deref-context*))
        (throw
         (ex-info
          "Reaction has already been disposed."
          {:rct rct})))
      rct)))

(defn one
  ([query-or-reaction] (one query-or-reaction 1000))
  ([query-or-reaction timeout-ms]
   (let [r (if (tidy/reaction? query-or-reaction)
             query-or-reaction
             (subscribe query-or-reaction))
         wait-ch (core-async/chan)]
     (tidy/subscribe
      r wait-ch
      {:include-initial? true
       :on-value (fn [value]
                   (when (tidy/initialized? r)
                     (core-async/close! wait-ch)))})
     (core-async/alts!!
      (cons wait-ch
            (when timeout-ms
              [(core-async/timeout timeout-ms)])))
     (let [value @r]
       (tidy/unsubscribe r wait-ch)
       value))))

(defn reaction->seq
  [reaction]
  (let [queue (LinkedBlockingQueue. 1)]
    (tidy/subscribe
     reaction queue
     {:include-initial? true
      :on-dispose (fn [] (.offer queue :tidy/complete-token))
      :on-value (fn [value] (.put queue value))})
    (concat
     (lazy-seq (.take queue))
     (queue->seq queue))))

;;; Adapters

(defn source
  "Note that this function _creates_ a source, and therefore has side effects.
  This function should then never be called from within the context of a
  reaction, but instead should be the result of a subscription (or more
  generally a cached reaction)."
  ([x] (source (str "source_" (gensym)) x))
  ([id x] (source id x nil))
  ([id x shutdown]
   (cond

     ;; Create a reaction whose only job is to return the value of an ratom
     ;; over time. The ratom's value is reset only once we have at least one
     ;; subscriber on the reaction we created (i.e. wait for subscription
     ;; before we start producing values...) This is guaranteed using the
     ;; control queue created with capacity 1.
     (instance? LinkedBlockingQueue x)
     (let [control-queue (LinkedBlockingQueue. 1)
           r (tidy/ratom)
           rct (tidy/reaction @r)]
       (tidy/on-state
        rct :tidy/started
        #(do
           (tidy/subscribe
            rct control-queue
            {:on-dispose
             (fn []
               (when (fn? shutdown) (shutdown))
               (.offer x :tidy/complete))})
           (.offer control-queue :tidy/control-token)))
       (future
         (.take control-queue)
         (try (loop []
                (let [value (.take x)]
                  (when-not (= value :tidy/complete)
                    (when-not (tidy/disposed? rct)
                      (reset! r value)
                      (recur)))))
              (catch Exception e
                (println e "Exception occurred in source loop"))))
       rct)

     (tidy/reaction? x) x

     ;; Create a reaction whose only job is to return the value of an ratom
     ;; over time. The ratom's value is reset only once we have at least one
     ;; subscriber on the reaction we created (i.e. wait for subscription
     ;; before we start producing values...) Backpressure is maintained using
     ;; a control queue with capacity 1.
     (tidy/ratom? x)
     (let [control-queue (LinkedBlockingQueue. 1)
           r (tidy/ratom)
           rct (tidy/reaction @r)]
       (tidy/on-state
        rct :tidy/started
        #(do (tidy/subscribe rct control-queue {:on-dispose (fn [] (when (fn? shutdown) (shutdown)))})
             (.offer control-queue :tidy/control-token)))
       (tidy/subscribe
        x rct
        {:on-value (fn [value]
                     (.take control-queue)
                     (when-not (tidy/disposed? rct)
                       (reset! r value)
                       (.put control-queue :tidy/control-token)))
         :include-initial? true})
       rct)

     :else (throw (ex-info "Unknown source type" {:x x})))))

(defn reg-source
  ([id source-fn] (reg-source id source-fn identity))
  ([id source-fn xform]
   (let [id-str (keyword->string id)
         make-id (comp keyword (partial str id-str))
         source-sub-id (make-id "-source-sub")
         sub-reaction-id (make-id "-ext-sub-rct")
         outer-sub-id id]
     (reg-sub-raw
      source-sub-id
      (fn [ctx params]
        (source-fn ctx params)))
     (reg-sub-raw
      outer-sub-id
      (fn [ctx params]
        (tidy/make-reaction
         sub-reaction-id
         (fn []
           (xform
            @(subscribe
              (vec (cons source-sub-id (rest params))))))))))))

;;; Dev Utilities

(defn clear-handlers! []
  (locking monitor
    (reset! handlers {})))

(defn clear-cache! []
  (locking monitor
    (try
      (doseq [[_ r] @query->reaction]
        (tidy/dispose r))
      (catch Exception e
        (log/warn e "Exception occurred clearing cache")))
    (reset! query->reaction {})))

(defn clear-all! []
  (clear-handlers!)
  (clear-cache!))

;;; Private

(defn- keyword->string
  [kw]
  (str (when-let [ns (namespace kw)]
         (str ns "/"))
       (name kw)))

(defn- queue->seq
  [queue]
  (let [value (.take queue)]
    (when-not (= :tidy/complete-token value)
      (cons value (queue->seq queue)))))
