(ns tidy.core
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]]))
  (:require

   ;; NOTE: as per discussion here: https://github.com/google/closure-compiler/issues/2336
   ;; we can't have a variable named 'async', so I have used the full name 'core-async'
   ;; for now.
   [clojure.core.async :as core-async :refer [ #?(:clj go) ]]

   #?(:clj [clj-time.core :as t])
   #?(:clj [clj-time.coerce :as tc])

   #?(:cljs [cljs-time.core :as t])
   #?(:cljs [cljs-time.coerce :as tc])))

;;; Declarations

(def ^:private default-interval-ms 1000)
(def ^:private channel-type (type (core-async/chan)))

;;; API

(defn channel?
  "Returns true when 'x' is a core.async channel instance."
  [x]
  (instance? channel-type x))

(defn interval-pipe
  "Places items from the 'from' channel onto the 'to' channel every
  't' ms from the last hand off."
  ([from to] (interval-pipe from to default-interval-ms))
  ([from to t]
   (assert (pos? t))
   (go
     (loop [last-hand-off-ms (t/minus (t/now) (t/millis t))]
       (when-let [v (core-async/<! from)]
         (let [t* (- (+ (tc/to-long last-hand-off-ms) t)
                     (tc/to-long (t/now)))]
           (core-async/<! (core-async/timeout t*))
           (core-async/>! to v)
           (recur (t/now)))))
     (core-async/close! to))))

(defn interval-pub
  "Works the same as a core.async 'pub', with the added functionality that no
  single subscription will receive a message more frequently than every 't' ms."
  ([ch topic-fn] (interval-pub ch topic-fn default-interval-ms))
  ([ch topic-fn t] (interval-pub ch topic-fn t (constantly nil)))
  ([ch topic-fn t buf-fn]
   (let [inner-pub-ch (core-async/chan)
         interval-channels (atom {})
         p (core-async/pub inner-pub-ch topic-fn buf-fn)
         topic-ch (fn [v]
                    (when-let [topic (topic-fn v)]
                      (or (get @interval-channels topic)
                          (let [from-ch (core-async/chan)]
                            (interval-pipe from-ch inner-pub-ch t)
                            (swap! interval-channels assoc topic from-ch)
                            from-ch))))

         unsub-all! (fn [topic]
                      (doseq [[topic* interval-ch] @interval-channels]
                        (when (or (nil? topic) (= topic topic*))
                          (core-async/close! interval-ch)
                          (swap! interval-channels dissoc topic*))))

         p* (reify
              core-async/Mux
              (muxch* [_] (core-async/muxch* p))
              core-async/Pub
              (sub* [_ topic ch close?]
                (core-async/sub* p topic ch close?))
              (unsub* [_ topic ch]
                (core-async/unsub* p topic ch))
              (unsub-all* [_]
                (unsub-all! nil)
                (core-async/unsub-all* p))
              (unsub-all* [_ topic]
                (unsub-all! topic)
                (core-async/unsub-all* p topic)))]

     (go
       (loop []
         (when-let [v (core-async/<! ch)]
           (when-let [ch* (topic-ch v)]
             (core-async/>! ch* v))
           (recur)))
       (unsub-all! p*)
       (core-async/close! inner-pub-ch))

     p*)))

(defn feeder-pipe
  "Move messages from 'from' to 'to', subject to being grouped by function 'f'.
  Messages will be placed into a queue with messages that produce the same
  result from 'f'. The queues will then be read, one at a time,
  non-deterministically and the items will be placed onto the 'to' channel."
  ([from to f] (feeder-pipe from to f 1024))
  ([from to f buf-size]
   (let [silos (atom {})
         stop-ch (core-async/chan)]

     ;; place items from 'from' channel onto the corresponding silo.
     (go
       (try (loop []
              (when-let [v (core-async/<! from)]
                (let [k (f v)]
                  (swap! silos
                         (fn [silos*]
                           (let [ch (or (get silos* k) (core-async/chan buf-size))]
                             (core-async/put! ch v)
                             (assoc silos* k ch))))
                  (recur))))
            (catch #?(:cljs js/Error) #?(:clj Exception) e
                (println e)))
       (doseq [ch (vals @silos)]
         (core-async/close! ch))
       (core-async/close! stop-ch)
       (core-async/close! to))

     ;; read non-deterministically from 'silos', placing results on 'to' channel.
     (go
       (try (loop []
              (let [[v ch] (core-async/alts!
                            (vec
                             (concat
                              [stop-ch (core-async/timeout 100)]
                              (vals @silos))))]
                (when (not= ch stop-ch)
                  (when (and v ch) (core-async/>! to v))
                  (recur))))
            (catch #?(:cljs js/Error) #?(:clj Exception) e
                (println e)))))))

;;; Private
