(ns tidy.subs
  (:require [tidy.core :as tidy]
            [clojure.core.async :as core-async]
            [utilis.fn :refer [fsafe]]
            [utilis.map :refer [compact map-vals]]
            [taoensso.timbre :as log])
  (:import [java.util.concurrent LinkedBlockingQueue]))

(declare keyword->string)

(defonce query->reaction (atom {}))
(defonce handlers (atom {}))
(def monitor (Object.)) ;; locking for now just to get it working

(defn clear-cache-for-id!
  [id]
  (swap! query->reaction
         (comp (partial into {})
            (partial remove (fn [[[id* & _] _]] (= id* id))))))

(defn reg-sub-raw
  [id f]
  (locking monitor
    (clear-cache-for-id! id)
    (swap!
     handlers assoc id
     {:f (fn [ctx query]
           (let [r (f ctx query)]
             (when (not (tidy/reaction? r))
               (throw
                (ex-info "Result of computation parameter in 'reg-sub-raw' must be a reaction"
                         {:id id
                          :r r})))
             r))})))

(defn subscribe
  [query]
  (let [{:keys [reaction new?]}
        (locking monitor
          (or (when-let [reaction (get @query->reaction query)]
                (when-not (tidy/disposed? reaction)
                  {:reaction reaction :new? false}))
              (if-let [handler-fn (:f (get @handlers (first query)))]
                (let [reaction (handler-fn nil query)]
                  (swap! query->reaction assoc query reaction)
                  {:reaction reaction :new? true})
                (throw (ex-info "No handler registered for query" {:query query})))))]
    (when (tidy/disposed? reaction)
      (throw
       (ex-info
        "Reaction has already been disposed."
        {:reaction reaction})))

    (when new?
      (let [listener-id (str "tidy/subs.subscribe." query)]
        (tidy/subscribe
         reaction listener-id
         {:silent? true
          :on-dispose
          (fn []
            (when (= reaction (get @query->reaction query))
              (swap! query->reaction dissoc query)))})))

    reaction))

(defn one
  ([query-or-reaction] (one query-or-reaction 1000))
  ([query-or-reaction timeout-ms]
   (let [r (if (tidy/reaction? query-or-reaction)
             query-or-reaction
             (subscribe query-or-reaction))
         wait-ch (core-async/chan)]
     (if (tidy/initialized? r)
       @r
       (do (tidy/subscribe
            r wait-ch
            {:on-value
             (fn [value]
               (core-async/close! wait-ch))})
           (core-async/alts!!
            (cons wait-ch
                  (when timeout-ms
                    [(core-async/timeout timeout-ms)])))
           (let [value @r]
             (tidy/unsubscribe r wait-ch)
             value))))))

;;; Adapters

(defn adapt-source
  ([source-fn] (adapt-source source-fn nil))
  ([source-fn on-dispose]
   (let [control-queue (LinkedBlockingQueue. 1)
         r (tidy/ratom)
         a (atom :tidy/none)
         rct (tidy/reaction @r)]
     (add-watch
      a control-queue
      (fn [_ _ _ value]
        (.take control-queue)
        (when-not (tidy/disposed? rct)
          (reset! r value)
          (.offer control-queue :tidy/control-token))))
     (let [source (source-fn a)]
       (tidy/subscribe
        rct control-queue
        {:silent? true
         :on-dispose (fn [] (when (fn? on-dispose) (on-dispose source)))
         :on-start (fn [] (.offer control-queue :tidy/control-token))}))
     rct)))

(defn reg-source
  ([id source-fn] (reg-source id source-fn nil))
  ([id source-fn on-dispose]
   (let [id-str (keyword->string id)
         source-sub-id ((comp keyword (partial str id-str)) "-source-sub")
         outer-sub-id id]
     (reg-sub-raw
      source-sub-id
      (fn [ctx params]
        (adapt-source
         #(source-fn (assoc ctx :r %) params)
         on-dispose)))
     (reg-sub-raw
      outer-sub-id
      (fn [ctx params]
        (tidy/reaction
         @(subscribe
           (vec (cons source-sub-id (rest params))))))))))

;;; Dev Utilities

(defn clear-handlers! []
  (locking monitor
    (reset! handlers {})))

(defn clear-cache! []
  (locking monitor
    (doseq [[_ r] @query->reaction]
      (try
        (tidy/dispose r)
        (catch Exception e
          (println e "Exception occurred disposing reaction"))))
    (reset! query->reaction {})))

(defn clear-all! []
  (clear-handlers!)
  (clear-cache!))

;;; Private

(defn- keyword->string
  [kw]
  (if (keyword? kw)
    (str (when-let [ns (namespace kw)]
           (str ns "/"))
         (name kw))
    (str kw)))
