(ns common.coll
  "Collection operations.
  >> ctdean jimbru"
  (:import common.KeyNotFoundException))

(declare keys-set)

(defn contains-every?
  "Tests whether a map contains each of the specified keys."
  [coll key & keys]
  (every? (partial contains? coll) (cons key keys)))

(defn contains-exactly?
  "Tests whether a map contains exactly the specified keys."
  [coll keys]
  (= (keys-set coll) (set keys)))

(defn ensure-coll
  "Ensure that X is a collection. If X is nil when return an empty collection."
  [x]
  (cond
   (coll? x) x
   (nil? x) []
   :else [x]))

(defn find-if
  "Returns the first item in coll for which (pred item) returns true.
  Note that this is subtley different than some, which returns the output
  of (pred item), rather than the item itself."
  [pred coll]
  (first (filter pred coll)))

(defn get-or-throw
  "Strict version of 'get' that throws when the key cannot be found."
  [map key]
  (when-not (contains? map key)
    (throw (KeyNotFoundException.)))
  (get map key))

(defn keys-set
  "Returns a map's keys as a set."
  [coll]
  (set (keys coll)))

(defn singleton?
  "True if the collection contains only a single element."
  [coll]
  (boolean (and (seq coll) (not (next coll)))))

(defn update-vals-if-exist
  "Update the values if the keys exist."
  [map keys f]
  (when map
    (reduce (fn [m key]
                (if (contains? m key)
                    (update-in m [key] f)
                    m))
        map
        keys)))

(defn best-by
  "Pick the 'best' element from the collection. Similar to `max-key`
  but works on non-numeric elements.

  `is-better?` is a function of two arguments: The current best
  element and the next element from the collection. It should return a
  boolean value.  To get the `worst' element, just negate the result."
  [is-better? coll]
  (when (seq coll)
    (reduce (fn [best x]
              (if (is-better? best x)
                  best
                  x))
            coll)))
