(ns common.sys)

(defn sleep
  "Sleep this thread for N milliseconds"
  [milliseconds]
  (Thread/sleep milliseconds))

(defn retry-action
  "Retry action ATTEMPTS times until SUCCESS? is achieved.
   Run PAUSE to wait between actions.

   Spec argument should be a map containing the following keys:

     action   - The action to be tried. 1-arity function where argument is current state.
     success? - [Opt] Function to execute on success. 1-arity with final state as arg.
     attempts - Integer number of retry attempts.
     initial-state - [Opt] Defaults to nil.
     pause    - [Opt] 2-arity function defining what to do between attempts. Defaults
                to sleeping the thread for at least 100 ms.
     finish   - [Opt] Action to run on completion. 2-arity function taking attempt number
                and state, respectively. Defaults to returning state.
   "
  [spec]
  (let [action (:action spec)
        success? (get spec :success? identity)
        attempts (get spec :attempts 5)
        initial-state (:initial-state spec)
        pause (get spec :pause (fn [n state] (sleep (* 100 n))))
        finish (get spec :finish (fn [n state] state))]
    (loop [i 0
           state initial-state]
      (when (< i attempts)
        (let [xstate (action state)]
          (if (success? xstate)
              (finish (inc i) xstate)
              (do
                (pause i xstate)
                (recur (inc i) xstate))))))))
