(ns common.edntoken
  "Create a a signed token encoded as an EDN map. Much simpler than JWT"
  (:refer-clojure :rename {read core-read})
  (:require
   [common.coding :refer [->b64-string b64-string->ba]]
   [byte-streams]
   [caesium.crypto.sign :as sign]
   [clojure.edn :as edn]))

(defn make
  "Create a signed auth token using a private key. DATA is an EDN
   object (a map probably) and we add an expiration time to the final
   token. Uses libsodium for all the signing."
  [data expiration-seconds secret-key]
  (let [now (quot (System/currentTimeMillis) 1000)
        payload {:expires (+ now expiration-seconds)
                 :data data}
        msg (pr-str payload)]
    (-> msg
        (byte-streams/to-byte-array)
        (sign/signed secret-key)
        (->b64-string))))

(defn- token->payload [token public-key]
  (-> token
      (b64-string->ba)
      (sign/verify public-key)
      (byte-streams/to-string)
      (edn/read-string)))

(defn read
  "Convert a token back to an EDN object (map). If the token has
   expired, return nil. If the token is invalid, throw an exception."
  [token public-key]
  (when token
    (let [payload (token->payload token public-key)
          now (quot (System/currentTimeMillis) 1000)]
      (when (>= (:expires payload) now)
        (:data payload)))))

(defn new-pk []
  (sign/keypair!))

(defn new-pk-as-strings []
  (-> (sign/keypair!)
      (update :public ->b64-string)
      (update :secret ->b64-string)))
