(ns hive.core.domain.board)
(require '[hive.core.domain.position :as position])
(require '[hive.core.domain.piece :as piece])

; board
;   for dealing with hive board states
;   querying the board and moving pieces about 

(def origin (position/create 0 0))

(def create {:pieces {}})

(defn purge-empties [pieces]
  (into {} (filter #(not (empty? (second %))) pieces)) )

(defn place-piece [board piece position]
  (cond
    (and board piece position)
      (update-in board [:pieces position] conj piece)
    :else
      board) )

(defn remove-piece [board position]
  (cond
    (and board position)
      (-> board
        (update-in [:pieces position] pop)
        (update-in [:pieces] purge-empties) )
    :else
      board) )

(defn move-piece [board position-0 position-1]
  (cond
    (and board position-0 position-1)
      (let [
        {{stack position-0} :pieces} board
        piece (last stack)]
        (-> board
          (remove-piece position-0)
          (place-piece piece position-1) ))
    :else
      board) )

(defn count-pieces 
  ([board]
    (->> board :pieces vals (map count) (reduce +)))
  ([board color-filter type-filter]
    (let [piece-predicate #(piece/is? % color-filter type-filter)
          filter-pieces #(filter piece-predicate %)]
      (->> board :pieces vals (map filter-pieces) (map count) (reduce +)) )))

(defn search-pieces [board color-filter type-filter]
  (->> board
    :pieces
    (map (fn [board-position]
      (let [position (first board-position)
            stack (second board-position)]
        (map-indexed (fn [index piece]
          {:position position, :layer index, :piece piece})
          stack) )))
    first
    (filter #(piece/is? (:piece %) color-filter type-filter)) ))

(defn search-top-pieces [board color-filter type-filter]
  (->> board
    :pieces
    (map (fn [board-position]
      (let [position (first board-position)
            stack (second board-position)]
        {:position position, :layer (->> stack count dec), :piece (last stack) }) ))
    (filter #(piece/is? (:piece %) color-filter type-filter)) ))

(defn lookup-occupied-positions [board]
  (keys (:pieces board)) )

(defn lookup-piece-stack [board position]
  (position (:pieces board)) )

(defn lookup-piece-stack-height [board position]
  (count (lookup-piece-stack board position)) )

(defn lookup-piece [board position]
  (last (lookup-piece-stack board position)) )

(defn lookup-piece-at-height [board position height]
  ((lookup-piece-stack board position) height) )

(defn lookup-adjacent-positions [board position]
  (zipmap
    position/directions-vector
    (map #((let [adjacent-position (position/translation %)] {
      :direction %
      :position adjacent-position
      :contents (lookup-piece-stack board adjacent-position)
    })) position/directions-vector) ))

; keys in this lookup table are specified as follows:
;   - keys have one character for each of six directions
;   - character order corresponds to position/directions-vector
;   - the sequence begins with 12 o'clock and proceeds clockwise
;   - the characters represent the contents of the position
;       one unit of distance away from an origin piece in the associated direction
;   - the character will be "1" if that direction is occupied
;   - the character will be "." if that direction is NOT occupied
; values in this lookup table correspond to the keys as follows:
;   - values have one character for each of six directions
;   - the character will be "1" if that direction is valid to slide into, given the occupied adjacencies
;   - the character will be "." if that direction is NOT valid to slide into, given the occupied adjacencies
;
; TODO: there must be a function that describes this data more compactly
; OBSERVATION 1: there are repeats in this data, configurations that are merely rotations of existing data
; OBSERVATION 2: this data need not be represented with strings, but could be any data structure
; OBSERVATION 3: an input could be rotated up to 5 times to compare to existing configurations for a match, and then rotated back
; OBSERVATION 4: some of the data could be generated from existing data
; OBSERVATION 5: there is an underlying rule that was used to generate this data,
;   dealing with the fact that a "gap" must be of a certain width in order to allow a slide to take place
;   perhaps the answer is to measure the gap, and compare that against the predicted width of the shape that must pass through it
;   since there are only 6 slide directions, this seems like it might be more clean than listing every possible configuration
;   plus, I'm not actually 100% certain that all of these are correct; I believe that they're probably at least 95% correct
;   but since they were generated by a human (me), and not fully tested, errors might still be present
(def can-slide-lookup-table {
  "......" "......" ; island cannot move
  ".....1" "1...1." ; slide around single piece
  "....1." "...1.1" ; slide around single piece
  "....11" "1..1.." ; slide alongside pair of adjacent pieces
  "...1.." "..1.1." ; slide around single piece
  "...1.1" "1.1..." ; slide up and out of crater
  "...11." "..1..1" ; slide alongside pair of adjacent pieces
  "...111" "1.1..." ; slide up and out of crater
  "..1..." ".1.1.." ; slide around single piece
  "..1..1" "11.11." ; slide between friends
  "..1.1." ".1...1" ; slide up and out of crater
  "..1.11" "11...." ; slide out of corner
  "..11.." ".1..1." ; slide alongside pair of adjacent pieces
  "..11.1" "11...." ; slide out of corner
  "..111." ".1...1" ; slide up and out of crater
  "..1111" "11...." ; slide to escape from pit
  ".1...." "1.1..." ; slide around single piece
  ".1...1" "..1.1." ; slide up and out of crater
  ".1..1." "1.11.1" ; slide between friends
  ".1..11" "..11.." ; slide out of corner
  ".1.1.." "1...1." ; slide up and out of crater
  ".1.1.1" "......" ; nearly-surrounded piece cannot move
  ".1.11." "1....1" ; slide out of corner
  ".1.111" "......" ; nearly-surrounded piece cannot move
  ".11..." "1..1.." ; slide alongside pair of adjacent pieces
  ".11..1" "...11." ; slide out of corner
  ".11.1." "1....1" ; slide out of corner
  ".11.11" "......" ; nearly-surrounded piece cannot move
  ".111.." "1...1." ; slide up and out of crater
  ".111.1" "......" ; nearly-surrounded piece cannot move
  ".1111." "1....1" ; slide to escape from pit
  ".11111" "......" ; nearly-surrounded piece cannot move
  "1....." ".1...1" ; slide around single piece
  "1....1" ".1..1." ; slide alongside pair of adjacent pieces
  "1...1." ".1.1.." ; slide up and out of crater
  "1...11" ".1.1.." ; slide up and out of crater
  "1..1.." ".11.11" ; slide between friends
  "1..1.1" ".11..." ; slide out of corner
  "1..11." ".11..." ; slide out of corner
  "1..111" ".11..." ; slide to escape from pit
  "1.1..." "...1.1" ; slide up and out of crater
  "1.1..1" "...11." ; slide out of corner
  "1.1.1." "......" ; nearly-surrounded piece cannot move
  "1.1.11" "......" ; nearly-surrounded piece cannot move
  "1.11.." "....11" ; slide out of corner
  "1.11.1" "......" ; nearly-surrounded piece cannot move
  "1.111." "......" ; nearly-surrounded piece cannot move
  "1.1111" "......" ; nearly-surrounded piece cannot move
  "11...." "..1..1" ; slide alongside pair of adjacent pieces
  "11...1" "..1.1." ; slide up and out of crater
  "11..1." "..11.." ; slide out of corner
  "11..11" "..11.." ; slide to escape from pit
  "11.1.." "....11" ; slide out of corner
  "11.1.1" "......" ; nearly-surrounded piece cannot move
  "11.11." "......" ; nearly-surrounded piece cannot move
  "11.111" "......" ; nearly-surrounded piece cannot move
  "111..." "...1.1" ; slide up and out of crater
  "111..1" "...11." ; slide to escape from pit
  "111.1." "......" ; nearly-surrounded piece cannot move
  "111.11" "......" ; nearly-surrounded piece cannot move
  "1111.." "....11" ; slide to escape from pit
  "1111.1" "......" ; nearly-surrounded piece cannot move
  "11111." "......" ; nearly-surrounded piece cannot move
  "111111" "......" ; completely surrounded piece cannot move
})

(defn encode-slide-lookup-key-from-adjacencies [position-adjacencies]
  (apply str (map #(if (nil? (:contents %)) \. \1 ) position-adjacencies) ))

; position/directions-vector
(defn render-valid-positions-from-slide-lookup-val [slide-lookup-val origin-position]
  (->> (map-indexed 
    (fn [idx dir] (let [is-valid (= \1 (nth slide-lookup-val idx))] 
      [dir is-valid] )) position/directions-vector)
    (filter #(second %))
    (map #(position/translation origin-position (first %))) ))

(defn pretend-position-is-empty [position-adjacencies empty-position]
  (map #(if (= (:position %) empty-position) 
    (update-in % [:contents] nil) 
    %) position-adjacencies) )

(defn lookup-adjacent-slide-positions 
  ([board position]
    (-> (lookup-adjacent-positions board position)
      encode-slide-lookup-key-from-adjacencies
      can-slide-lookup-table
      (render-valid-positions-from-slide-lookup-val position) ))
  ([board position empty-position]
    (-> (lookup-adjacent-positions board position)
      (pretend-position-is-empty empty-position)
      encode-slide-lookup-key-from-adjacencies
      can-slide-lookup-table
      (render-valid-positions-from-slide-lookup-val position) )) )



