(ns tulos.commons.ts
  (:require [clojure.java.io :as io]
            [clojure.spec.alpha :as s]
            [clojure.string :as string]
            [influx.lp.encode :as i.e]
            [java-time :as j]
            [cheshire.core :as json]
            [net.cgrand.xforms :as x]))

(s/def ::named (s/or :named #(instance? clojure.lang.Named %)
                     :str string?))

(s/def ::tag-key keyword?)
(s/def ::measurement keyword?)
(s/def ::tag-value string?)
(s/def ::field keyword?)
(s/def ::value double?)
(s/def ::inst inst?)
(s/def ::local-date j/local-date?)

(s/def ::point-time
  (s/or :daily ::local-date
        :inst ::inst))

(s/def ::tagset
  (s/map-of ::tag-key ::tag-value))
(s/def ::values
  (s/map-of ::field ::value))
(s/def ::points
  (s/map-of ::point-time ::values))

;; A Batch is a map of tagset -> points:
;;
;;   {{:symbol "futures/wti-nymex/z17",
;;     :cps "wti-nymex",
;;     :contract-period "z17",
;;     :instrument-type "futures"}
;;    {#inst "2017-04-28T10:42:02.334-00:00"
;;     {:bid 50.7, :volume-bid 3, :ask 50.71, :volume-ask 1},
;;     #inst "2017-04-28T10:42:04.702-00:00"
;;     {:ask 50.7, :volume-ask 12, :bid 50.69, :volume-bid 4}}
;;
;; All the points in the same batch must have the same timestamp granularity
(s/def ::batch
  (s/map-of ::tagset ::points))

;; To Influx

(defn- map->str [m esc-k esc-v]
  (let [join-kv (fn [[k v]]
                  (str (esc-k k) "=" (esc-v v)))]
    (->> (map join-kv m)
         (sort)
         (string/join ","))))

(defn- ns->precision [precision ns]
  (case precision
    :ns ns
    :u (long (/ ns 1000))
    :ms (long (/ ns 1000 1000))
    :s (long (/ ns 1000 1000 1000))
    :m (long (/ ns 1000 1000 1000 60))
    :h (long (/ ns 1000 1000 1000 60 60))))

(defn- ms->precision [precision ms]
  (case precision
    :ns (* ms 1000 1000)
    :u (* ms 1000)
    :ms ms
    :s (long (/ ms 1000))
    :m (long (/ ms 1000 60))
    :h (long (/ ms 1000 60 60))))

(defn- sec->precision [precision sec]
  (case precision
    :ns (* sec 1000 1000 1000)
    :u (* sec 1000 1000)
    :ms (* sec 1000)
    :s sec
    :m (long (/ sec 60))
    :h (long (/ sec 60 60))))

(defn- date->long [precision ts]
  (cond
    (inst? ts) (ms->precision precision (.getTime ^java.util.Date ts))

    (j/local-date? ts)
    (->> (java.time.ZoneId/of "UTC")
         (.atStartOfDay ^java.time.LocalDate ts)
         (.toEpochSecond)
         (sec->precision precision))))

(s/def :ts.influx/measurement ::named)
(s/def :ts.influx/tags (s/map-of ::named ::named))
(s/def :ts.influx/value
  (s/or ::double double?
        ::int int?
        ::string string?
        ::boolean boolean?))
(s/def :ts.influx/fields (s/map-of ::named :ts.influx/value))
(s/def :ts.influx/timestamp
  (s/or :inst inst?
        :local-date j/local-date?
        :epoch-ns int?))

(s/def ::tags-with-measurement :ts.influx/tags)
(s/def ::timestamped-points
  (s/map-of :ts.influx/timestamp :ts.influx/fields))
(s/def ::tagset->timestamped-point-batch
  (s/map-of ::tags-with-measurement ::timestamped-points))

(s/fdef timestamped-batch->lines
        :args (s/cat :batch ::tagset->timestamped-point-batch
                     :ts-fn fn?
                     :measurement-key ::named)
        :ret string?)

(defn timestamped-batch->lines [point-batch ts->long measurement-key]
  (let [^StringBuilder sb (StringBuilder. (count point-batch))
        tags->str #(map->str (dissoc % measurement-key) i.e/tag-key i.e/tag-value)
        fields->str #(map->str % i.e/field-key i.e/field-value)]

    (loop [[tagset ts-points] (first point-batch), more (rest point-batch)]
      (when tagset
        (let [m (i.e/measurement (get tagset measurement-key))
              tag-str (tags->str tagset)
              tag-part (str m (if (seq tag-str) (str "," tag-str) tag-str))
              cnt (count ts-points)]

          (loop [[ts fields] (first ts-points), more (rest ts-points)]
            (when ts
              (let [field-str (fields->str fields)
                    ts-str (ts->long ts)]
                (.append sb (str tag-part " " field-str " " ts-str)))
              (when (seq more)
                (.append sb "\n")
                (recur (first more) (rest more))))))

        (when (seq more)
          (.append sb "\n")
          (recur (first more) (rest more)))))

    (.toString sb)))

(defn ts-batch->lines
  "Writes the `point-batch` where timestamps are numbers of whichever precision
  is specified in the influx write parameters. Timestamps are sent to Influx
  the same way they are represented in the `point-batch`.

    * ts = `412008`, precision = `:h`"
  [point-batch measurement-key]
  (timestamped-batch->lines point-batch
    identity measurement-key))

(defn date-batch->lines-fn
  "Given a valid Influx `precision` (ns, ms, u, s, m, h) creates a batch-writing
  function which will write the `point-batch` where timestamps are date objects.
  Timestamp dates are converted to epoch numbers in the specified `precision`.

  For example, for the time of 2017-01-01T00:00:00 UTC we can have:

    * ts = `1483228800`, precision = `:s`
    * ts = `24720480`, precision = `:m`
    * ts = `412008`, precision = `:h`"
  [precision]
  (fn [point-batch measurement-key]
    (timestamped-batch->lines point-batch
      #(date->long precision %) measurement-key)))

;; From influx

(defn time->instant
  "Given a valid Influx `precision` and a `ts` (timestamp) number in that
  precision, creates an instant object."
  [precision ts]
  (j/instant
    (case precision
      :ns (/ ts 1000 1000)
      :u (/ ts 1000)
      :ms ts
      :s (* ts 1000)
      :m (* ts 1000 60)
      :h (* ts 1000 60 60))))

(defn time->local-date
  "Given a valid Influx `precision` and a `ts` (timestamp) number in that
  precision, creates a local date in the UTC timezone."
  [precision ts]
  (-> (time->instant precision ts)
      (j/local-date "UTC")))

(defn series->timestamped-batch [time-fn {:keys [columns values partial]}]
  (when partial
    (throw (ex-info "Got partial result!" {:name name, :columns columns})))
  (let [k-columns (map keyword columns)]
    (into {}
          (comp (map #(zipmap k-columns %))
                (map (fn [{:keys [time] :as all}]
                       [(time-fn time) (dissoc all :time)])))
          values)))

(defn influx-result->statement-tags-map
  "Transforms the `result-json` (parsed into EDN and keywordized)
  into a nested structure of: `[StatementIdx [Tags [TS Fields]]`.

  This only works if the results are grouped by tags."
  [result-json series-tf]
  (into {}
    (x/by-key :statement_id
      (comp (mapcat :series)
            (x/by-key :tags
              (comp (map series-tf)
                    (x/into {})))
            (x/into {})))
    (:results result-json)))

;; TODO:
;;  The chunked response returns a stream of json objects which needs splitting
;;  on newlines between the valid `result` blocks - easy, but fails currently as we don't even try that.
(defn response-stream->json [response-json-body]
  (let [parse #(json/parse-stream % true)]
    (-> response-json-body io/reader parse)))
