(ns tusk.adapter-clojure-jdbc
  (:require
   [tusk
    [core :as c]
    [util :as u]]
   [tusk.types
    [clojure :as clj]
    [array :as arr]
    [json :as json]
    [geometric :as g]
    [postgis :as gis]])
  (:import
   [org.postgresql.geometric PGbox PGcircle PGline PGlseg PGpath PGpoint PGpolygon]
   [org.postgis PGgeometry PGgeometryLW]
   [java.util UUID]
   [org.postgresql.jdbc PgArray]
   [org.postgresql.util PGobject]))

(defn ^:private coerce-children
  [data connection metadata index]
  (mapv #(jdbc.proto/from-sql-type % connection metadata index)
        data))

(->> (for [c arr/default-classes]
       `(extend ~c
          jdbc.proto/ISQLResultSetReadColumn
          {:from-sql-type coerce-children}))
     (cons 'do)
     (eval))

(extend-protocol jdbc.proto/ISQLResultSetReadColumn
  PGpoint
  (from-sql-type [d & _] (g/geom-edn d))

  PGcircle
  (from-sql-type [d & _] (g/geom-edn d))

  PGbox
  (from-sql-type [d & _] (g/geom-edn d))

  PGbox
  (from-sql-type [d & _] (g/geom-edn d))

  PGline
  (from-sql-type [d & _] (g/geom-edn d))

  PGlseg
  (from-sql-type [d & _] (g/geom-edn d))

  PGpath
  (from-sql-type [d & _] (g/geom-edn d))

  PGpolygon
  (from-sql-type [d & _] (g/geom-edn d))

  PGgeometryLW
  (from-sql-type [d & _] (gis/geo-edn d))

  PGgeometry
  (from-sql-type [d & _] (gis/geo-edn d))

  UUID
  (from-sql-type [d & _] d)

  PGobject
  (from-sql-type [d conn metadata idx]
    (c/read (.getValue d) {:from (.getType d)
                           :connection conn}))

  PgArray
  (from-sql-type [d conn metadata idx]
    (mapv #(jdbc.proto/from-sql-type % conn metadata nil)
          (.getArray d))))

(defn ^:private
  coerce-and-retry
  [this conn stmt index]
  (-> this
      (c/coerce {:to (u/parameter-name stmt index)})
      (jdbc.proto/set-stmt-parameter! conn stmt index)))

(defn ^:private
  throw-not-implemented
  [this conn]
  (throw (ex-data "cannot be cast to sql type"
                  {:casting this})))

(->> (for [c [clojure.lang.PersistentHashMap
              clojure.lang.PersistentArrayMap
              clojure.lang.PersistentList
              clojure.lang.PersistentList$EmptyList
              clojure.lang.PersistentHashSet]]
       `(extend ~c
          jdbc.proto/ISQLType
          {:set-stmt-parameter! coerce-and-retry
           :as-sql-type         throw-not-implemented}))
     (cons 'do)
     (eval))

(extend-protocol jdbc.proto/ISQLType

  clojure.lang.IPersistentVector
  (set-stmt-parameter! [this conn stmt index]
    (let [[arr? typ] (clj/vector-dispatch this stmt index)
          o          (if arr?
                       (arr/pg-array typ this)
                       (c/coerce this {:to typ}))]
      (jdbc.proto/set-stmt-parameter! o conn stmt index)))
  (as-sql-type [this conn]
    (throw (ex-data "cannot be cast to sql type"
                    {:casting this})))

  tusk.types.array.PgArray
  (set-stmt-parameter! [this conn stmt index]
    (->> (jdbc.proto/as-sql-type this conn)
         (.setArray stmt index)))
  (as-sql-type [{v   :vector
                 typ :pg-type} conn]
    (->> (arr/into-nested-array identity v)
         (.createArrayOf conn typ))))
