(ns tusk.tusk-core.util
  (:require
   [clojure.string :as str])
  (:import
   [org.postgresql.util
    PGobject]))

(defn statement-parameter-name
  [stmt idx]
  (some-> stmt
          (.getParameterMetaData)
          (.getParameterTypeName idx)))

(defn pg-object
  "turn a value into a postgres object of a specified type"
  [v pg-type]
  (when (and v pg-type)
    (doto (PGobject.)
      (.setType pg-type)
      (.setValue (name v)))))

(defn ^:private
  surround
  [s ends]
  (str ends s ends))

(defn snake
  [s]
  (when (and s (string? s))
    (str/escape s {\- \_})))

(defn kebab
  [s]
  (when (and s (string? s))
    (str/escape s {\_ \-})))

(defn ident-str
  "convert from a clojure keyword to a postgres identifier.
  snake-case is kebab-cased and the schema becomes the key's namespace"
  [s]
  (when s
    (if (string? s)
      s
      (if-let [parents (when (keyword? s)
                         (some-> (namespace s)
                                 (str/split #"\.")))]
        (-> (namespace s)
            (str/split #"\.")
            (conj (name s))
            (->> (map #(surround % \"))
                 (map snake)
                 (str/join ".")))
        (name s)))))

(defn ident-key
  "convert from a clojure keyword to a postgres identifier.
  kebab-case is snake-cased and the key's namesaces is taken as the schema"
  [k]
  (when k
    (if (keyword? k)
      k
      (if-let [path (->> (re-seq #"\"([^\"]+)\"\.?" k)
                         (map second))]
        (keyword (->> path butlast (map kebab) (str/join \.))
                 (->> path last kebab))
        (keyword k)))))

(defn hobs
  "Based on the passed predicate, the resulting function will
  return either the value passed or nil.

  like hobson's choice: take it or leave it"
  [pred]
  (fn [x]
    (when (pred x)
      x)))
