(ns uk.axvr.dynamock.http
  "HTTP mocking utilities for uk.axvr.dynamock."
  (:require [uk.axvr.dynamock :as mock]))

;; Copyright © 2016 Rich Hickey.
;; https://github.com/clojure/spec-alpha2/blob/74ada9d5111aa17c27fdef9c626ac6b4b1551a3e/src/test/clojure/clojure/test_clojure/spec.clj#L18,L25
(defn- submap?
  "Returns true if map1 is a subset of map2."
  [map1 map2]
  (if (and (map? map1) (map? map2))
    (every? (fn [[k v]]
              (and (contains? map2 k)
                   (submap? v (get map2 k))))
            map1)
    (= map1 map2)))

(defn stub-pred-matches?
  "Checks if an http-stub-predicate (pred) \"matches\" the parameters (params)
  passed to the mocked function and returns the stub."
  [params [pred stub]]
  (when
    (cond
      (fn? pred)   (apply pred params)
      (map? pred)  (submap? pred (first params))
      (coll? pred) (= pred params))
    stub))

(defn stub->resp
  "Get the response from a stub + params."
  [stub params]
  (if (fn? stub) (apply stub params) stub))

(defn ^:internal derefable?
  "Returns true if `clojure.core/deref` can be called on ref."
  [ref]
  #?(:cljs    (or (satisfies? IDeref ref)
                  (satisfies? IDerefWithTimeout ref))
     :default (or (instance? clojure.lang.IDeref ref)
                  (instance? clojure.lang.IBlockingDeref ref))))

(defn ^:internal ->derefable
  "Returns `x` as a \"derefable\" object (if it is not already derefable)."
  [x]
  (if (derefable? x) x (delay x)))

(defn ^:internal <-derefable
  "When `x` is a \"derefable\" object, it will be dereferenced returned,
  otherwise `x` will be returned."
  ([x]
   (if (derefable? x) @x x))
  ([x timeout-ms timeout-val]
   (if (derefable? x) (deref x timeout-ms timeout-val) x)))

(defonce
  ^{:doc "Atom containing a map of the default options for the
  `uk.axvr.dynamock.http/http-mock` function.

  See: `uk.axvr.dynamock.http/http-mock` for details on what options are
  available."}
  default-opts
  (atom {:transform-request  identity
         :transform-response (fn [_ resp] resp)}
        :validator map?))

(defn http-mock
  "Simple HTTP mock function generator.

  Accepts an optional map of config options (opts):

    :transform-request

      Function that is passed a list of parameters that were given to the mocked
      function.  Returns the input with any desired alterations made.

      Note that the original unmodified parameters will still be passed to the
      original base HTTP function if no matching stub was found.

      Default: `clojure.core/identity`

      Example: Parse JSON body into Clojure data for better stub selection.

        (fn [params]
          (update-in params [0 :body]
            cheshire.core/parse-string true))

    :transform-response

      Function to make final modifications to the mocked response before
      returning it.  It is passed the parameters given to the mocked function
      and the response.  It is expected to return the response.

      Default:

        (fn [params response] response)

      Example: behave like HttpKit, by making responses derefable.

        (fn [params response]
          (uk.axvr.dynamock.http/->derefable response))

  Use `uk.axvr.dynamock.http/default-opts` to set default options."
  ([real-fn get-stubs]
   (http-mock real-fn get-stubs {}))
  ([real-fn get-stubs opts]
   (let [{:keys [transform-request transform-response]} (merge @default-opts opts)]
     (fn [& params]
       (let [tformed-params (transform-request params)]
         (if-let [stub (some (partial stub-pred-matches? tformed-params) (get-stubs))]
           (transform-response
             tformed-params
             (stub->resp stub tformed-params))
           (apply real-fn params)))))))

(defmacro with-http-mock
  "Register the below scope with a mocked HTTP fn.

  If the first value in the body is a map (and there is more than one form in
  the body) it will be treated as options and passed to
  `uk.axvr.dynamock.http/http-mock`.

  See: `uk.axvr.dynamock.http/http-mock` for supported configuration options."
  [fn & body]
  (let [[opts body] (r/macro-body-opts body)]
    `(mock/with-mock ~fn #(http-mock %1 %2 ~opts)
       ~@body)))

(def block-real-http-requests
  "Stub to block real HTTP requests."
  [(constantly true)
   #(throw
      (ex-info "uk.axvr.dynamock: No matching stub found.  Real HTTP requests are blocked." %))])
