(ns uswitch.ring-riemann
  (:require [riemann.client]
            [clojure.string :as string]
            [clj-stacktrace.repl :as stacktrace]))

(defn mapply
  [f & args]
  (apply f (apply concat (butlast args) (last args))))

(defn- make-client
  [client-opts]
  (let [client-fn (if (= :udp (:protocol client-opts))
                    riemann.client/udp-client
                    riemann.client/tcp-client)]
    (mapply client-fn client-opts)))

(defn default-success-fn
  [service]
  (fn [req res time]
    {:service        service
     :state          (str (:status res))
     :metric         time}))

(defn default-exception-fn
  [service]
  (fn [req time throwable]
    {:service        service
     :state          "error"
     :tags           ["exception"]
     :metric         time
     :description    (stacktrace/pst-str throwable)
     :exception-type (str (type throwable))} ))

(defn wrap-riemann
  [handler client-opts success-fn exception-fn]
  (let [client (make-client client-opts)
        ack?   (get client-opts :ack false)]
    (fn [req]
      (let [t (System/currentTimeMillis)]
        (try
          (let [res (handler req)]
            (riemann.client/send-event client
                                       (success-fn req res (- (System/currentTimeMillis) t))
                                       ack?)
            res)
          (catch Throwable ex
            (riemann.client/send-event client
                                       (exception-fn req (- (System/currentTimeMillis) t) ex)
                                       ack?)
            (throw ex)))))))


(defn wrap-riemann-service
  [handler client-opts service]
  (wrap-riemann handler
                client-opts
                (default-success-fn service)
                (default-exception-fn service)))
