(ns ventas-devtools.client.auto
  "https://github.com/weavejester/lein-auto/blob/master/src/leiningen/auto.clj"
  (:require
   [clojure.java.io :as io]
   [clojure.string :as str]
   [taoensso.timbre :as timbre])
  (:import
   [java.io File]))

(defn directory-files [dir]
  (->> (io/file dir) (file-seq) (remove (memfn isDirectory))))

(defn modified-since [^File file timestamp]
  (> (.lastModified file) timestamp))

(defn grep [re coll]
  (filter #(re-find re (str %)) coll))

(defn add-ending-separator [^String path]
  (if (.endsWith path File/separator)
    path
    (str path File/separator)))

(defn remove-prefix [^String s ^String prefix]
  (if (.startsWith s prefix)
    (subs s (.length prefix))
    s))

(defn show-modified [project files]
  (let [root  (add-ending-separator (:root project))
        paths (map #(remove-prefix (str %) root) files)]
    (str/join ", " paths)))

(def default-config
  {:file-pattern #"\.(clj|cljs|cljx|cljc)$"
   :wait-time    50
   :log-color    :magenta})

(defn default-paths [project]
  (concat (:source-paths project)
          (:java-source-paths project)
          (:test-paths project)))

(defn auto
  "Executes the given task every time a file in the project is modified."
  [{:keys [project config identifier task-fn]}]
  (let [config (merge default-config
                      {:paths (default-paths project)}
                      (get-in project [:auto :default])
                      (get-in project [:auto identifier])
                      config)]
    (future
     (loop [time 0]
       (when-not (Thread/interrupted)
         (Thread/sleep (:wait-time config))
         (if-let [files (->> (mapcat directory-files (:paths config))
                             (grep (:file-pattern config))
                             (filter #(modified-since % time))
                             (seq))]
           (let [time (System/currentTimeMillis)]
             (timbre/info "Files changed:" (show-modified project files))
             (timbre/info "Running" identifier)
             (try
               (task-fn)
               (timbre/info "Completed.")
               (catch Throwable e
                 (timbre/error (.getMessage e))))
             (recur time))
           (recur time)))))))
