(ns ventas-devtools.client.sassc
  "https://github.com/apribase/lein-sassc/blob/master/src/leiningen/sassc.clj"
  (:refer-clojure :exclude [compile])
  (:require
   [clojure.java.shell :as shell]
   [clojure.string :as str]
   [clojure.string :as string]
   [me.raynes.fs :as fs]
   [taoensso.timbre :as timbre]))

(defn- create-config [node]
  (let [default {:src "src/scss/main.scss"
                 :output-to "target/sassc/main.css"
                 :style "nested"
                 :import-path "src/scss"}]
    (merge default node)))

(defn parse-import-path
  "Create the import path options for sassc, adding multiple options if the argument is a vector"
  [import-path]
  (let [import-path (if (vector? import-path) import-path [import-path])]
    (apply str (interleave (repeat " -I ") import-path))))

(defn- run-sassc-command
  "Run sassc command, compile a SASS/SCSS file."
  [config]
  (let [{:keys [src output-to style import-path]} config
        import-path (parse-import-path import-path)
        command (str "sassc -t " style import-path " " src " " output-to)]
    (timbre/debug command)
    (apply shell/sh (string/split command #"\s+"))))

(defn- compile-node [node]
  (let [config (create-config node)]
    (-> (:output-to config)
        (fs/parent)
        (fs/mkdirs))
    (run-sassc-command config)))

(defn compile
  "Compile the :sassc project once."
  [config]
  (doseq [node config]
    (as-> (compile-node node) %
          (case (:exit %)
            0 (let [out (:out %)]
                (when-not (str/blank? out)
                  (timbre/info out)))
            (let [err (:err %)]
              (when-not (str/blank? err)
                (throw (Exception. (str err)))))))))

(defn clean
  "Remove automatically generated files."
  [config]
  (timbre/info "Deleting CSS files generated by the SASS compiler")
  (doseq [node config]
    (fs/delete (:output-to node))))

