(ns ventas-devtools.uberjar
  (:require
   [ventas-devtools.config :as devtools.config]
   [ventas-devtools.client.sassc :as sassc]
   [ventas.logging]
   [cljs.build.api]
   [taoensso.timbre :as timbre]
   [me.raynes.fs :as fs]))

(defn minified-build [theme]
  (let [name (name theme)]
    {:id (str "min-" name)
     :source-paths ["src/cljs" "src/cljc" "custom-lib"]
     :compiler {:main (symbol (str "ventas.themes." name ".core"))
                :output-to (str "resources/public/files/js/compiled/" name ".js")
                :output-dir (str "resources/public/files/js/compiled/" name)
                :npm-deps {:js-image-zoom "0.5.0"}
                :install-deps true
                :source-map-timestamp true
                :optimizations :advanced
                :pretty-print false
                :externs ["externs.js"]
                :parallel-build true}}))

(defn- elapsed [started-at]
  (let [elapsed-us (- (System/currentTimeMillis) started-at)]
    (with-precision 2
      (str (/ (double elapsed-us) 1000) " seconds"))))

(defn- get-output-files [compiler-options]
  (if-let [output-file (:output-to compiler-options)]
    [output-file]
    (into [] (map :output-to (->> compiler-options :modules vals)))))

(defn compile-cljs [cljs-paths compiler-options & {:keys [assert?]}]
  (let [output-files (get-output-files compiler-options)
        output-files-parent (map fs/parent output-files)]
    (timbre/info (str "Compiling " (pr-str output-files) " from " (pr-str cljs-paths) "..."))
    (flush)
    (fs/delete-dir (:output-dir compiler-options))
    (doseq [output-file-parent output-files-parent]
      (when output-file-parent
        (fs/mkdirs output-file-parent)))
    (let [started-at (System/currentTimeMillis)]
      (try
        (binding [*assert* assert?]
          (cljs.build.api/build (apply cljs.build.api/inputs cljs-paths) compiler-options))
        (doseq [output-file output-files]
          (fs/touch output-file started-at))
        (timbre/info "Successfully compiled" (pr-str output-files) "in" (elapsed started-at) "." )
        (catch Throwable e
          (timbre/error "Compiling" (pr-str output-files) "failed."))))))

(defn prepare [config]
  (let [{:keys [themes]} (read-string config)]
    (timbre/info "Compiling SASS")
    (sassc/compile (devtools.config/sassc themes))
    (timbre/info "Compiling Clojurescript")
    (doseq [theme themes]
      (let [build (minified-build theme)]
        (compile-cljs (:source-paths build) (:compiler build) :assert? true)))))