(ns webaudio.microphone-stream
  (:use-macros [purnam.core :only [? ! !>]]
               [cljs.core.async.macros :only [go]])
  (:require [cljs.core.async :refer [<!]]
            [usermedia.core :refer [get-user-media get-user-media-by-label]]))

;TODO: make it an external namespace

(def stream nil)
(def analyser nil)
(def callback nil)
(def calibration-table nil)
(def LOG_10 (.log js/Math 10))
(def TEN_OVER_LOG_10 (/ 10 LOG_10))
(def constraints {:audio {:mandatory {:googAutoGainControl false 
                                      :googNoiseSuppression false 
                                      :googHighpassFilter false}}})

(def freq-cut-index nil)

(defn amplitude[arr]
  (areduce arr i res 0
    (if (< 0 i freq-cut-index)
      res
      (+ res (.pow js/Math 10 (/  (aget arr i) 10))))))

(defn sum [a b]
  (amap a i res (+ (aget b i) (aget a i))))

(defn linear-sq-to-db "10 * log10(x) : we use squared values => we multiply by 10 instead of 20"
  [val]
  (* TEN_OVER_LOG_10 (.log js/Math val)))

(defn calc-volume [valuesInDb]
  (-> valuesInDb
      amplitude
      linear-sq-to-db))

(defn update-values [audio-context]
  (let [array-fft-float (js/Float32Array. (.-frequencyBinCount analyser))
        _ (.getFloatFrequencyData analyser array-fft-float)
        calibrated (sum array-fft-float calibration-table)]
    (callback (.-currentTime audio-context) (calc-volume calibrated) calibrated)
    (js/requestAnimationFrame (partial update-values audio-context))))

(defn connect [mic-label onSuccess onError]
  (go
    (if stream
      (onSuccess)
      (let [[status data] (<! (if mic-label 
                                  (get-user-media-by-label (re-pattern mic-label) constraints)
                                  (get-user-media constraints)))]
        (case status
          :error (onError data)
          :ok (do (set! stream data) 
                  (onSuccess)))))))

(defn freq-min [sampling-rate fftSize]
  (/ sampling-rate fftSize))

(defn create-analyser [audio-context calibration-table-in fft-size smoothing-time-tonstant freq-cut cb after-init]
  (set! callback cb)
  (set! calibration-table calibration-table-in)
  (set! freq-cut-index (/ freq-cut (freq-min (? audio-context.sampleRate) fft-size)))
  (when-not analyser 
    (set! analyser (!> audio-context.createAnalyser))
    (! analyser.smoothingTimeConstant smoothing-time-tonstant)
    (! analyser.fftSize fft-size)
    (let [stream (!> audio-context.createMediaStreamSource stream)]
        (!> stream.connect analyser))
    (update-values audio-context))
  (when after-init
    (after-init (? analyser.frequencyBinCount) (? analyser.fftSize) (? audio-context.sampleRate))))
