(ns vise890.multistub.core
  "Utilities for stubbing multi-methods.")

(defn with-around-fns
  "Executes `f` within the context of `around-fns`. `around-fns` is a (sorted!)
  sequence of functions that take a no arg function.

  E.g.:

  (with-around-fns [with-zookeeper-fn
                    with-kafka-fn]
                   produce-some-stuff-on-kafka-fn)"
  [around-fns f]
  (cond
    (empty? around-fns) (f)
    (= 1 (count around-fns)) ((first around-fns) f)
    :else (with-around-fns (butlast around-fns)
            (fn [] ((last around-fns) f)))))

(defn- add-method
  "Installs a new method of `multifn` associated with `dispatch-valueue`."
  [multifn dispatch-value f]
  (.addMethod multifn dispatch-value f))

(defn with-multi-stub
  "Executes `f` within the original implementation of `multifn` for
  `dispatch-value` substituted with `stub-fn`.

  It restores the original implementation of `multifn` for `dispatch-value`
  before returning."
  [multifn dispatch-value stub-fn f]
  (let [original-method (get-method multifn dispatch-value)
        ;; TODO not needed?
        _ (remove-method multifn dispatch-value)
        _ (add-method multifn dispatch-value stub-fn)
        ret (f)
        _ (remove-method multifn dispatch-value)
        _ (add-method multifn dispatch-value original-method)]
    ret))

(defn with-multi-stubs
  "Executes `f` with `stubs` defined and then restores their original bindings.

  E.g.:

  (with-multi-stubs
    [my-multi-1 :dispatch-val-x (fn [_] :stub-x)
     my-multi-2 :dispatch-val-y (fn [_] :stub-y)]
   f)"
  [stubs f]
  (let [around-fns (->> stubs
                        (partition 3)
                        (map (fn [[multifn dispatch-value stub-f]]
                               (partial with-multi-stub multifn dispatch-value stub-f))))]
    (with-around-fns around-fns f)))

(defn with-dispatch-value-multi-stubs
  "Executes `f` with `stubs` defined on `dispatch-value` and then restores their
  original bindings.

  E.g.:

  (with-dispatch-value-multi-stubs :my-dispatch-val
    [my-multi-1 (fn [_] :stub-1)
     my-multi-2 (fn [_] :stub-2)]
   f)"
  [dispatch-value stubs f]
  (let [stubs (->> stubs
                   (partition 2)
                   (mapcat (fn [[multifn stub-f]]
                             [multifn dispatch-value stub-f])))]
    (with-multi-stubs stubs f)))
