(ns parseq.parsers
  "An eclectic bunch of parsers."
  (:require [parseq.utils :as pu]))

(defn ->fail
  "Builds a parser that always fails."
  ([msg] (fn [input] (pu/->failure msg)))
  ([msg data] (fn [input] (pu/->failure msg data))))

(def
  ^{:doc "A parser that always fails"}
  fail
  (->fail "`fail` failed, it always does that..."))

(defn one-satisfying
  "Builds a parser that takes an element that satisfies `predicate`."
  [predicate]
  (fn [input]
    (if-not (seq input)
      (pu/->failure "`one-satisfying` failed: no more input"
                    {:predicate predicate})
      (if-not (predicate (first input))
        (pu/->failure "`one-satisfying` failed: `predicate` returned false for `v`"
                      {:input     input
                       :predicate predicate
                       :v         (first input)})
        [(first input) (rest input)]))))

(def one
  "A parser that takes (any) one element."
  (one-satisfying (constantly true)))

(defn one-not-satisfying
  "Builds a parser that takes an element that does not satisfy `predicate`."
  [predicate]
  (one-satisfying (complement predicate)))

(defn one=
  "Builds a parser that takes an element that equals `v`."
  [v]
  (one-satisfying (partial = v)))

(defn one-not=
  "Builds a parser that takes an element that equals `v`."
  [v]
  (one-not-satisfying (partial = v)))
