(ns scribe.avro.core
  (:require [medley.core :refer [assoc-some filter-keys map-keys]]))

(defn- name+ [x] (when x (name x)))

(defn- disqualify-key [k]
  (-> k name keyword))

(defn- ns= [ns k]
  (= (name ns) (namespace k)))

;;,------------------------
;;| Avro Schema conversions
;;`------------------------
(defn- ->base-avro-schema [registry schema-name]
  (if-let [s       (get registry schema-name)]
    (let [avro-n  (name+ (or (:avro/name s)
                             (name schema-name)))
          avro-ns (name+ (or (:avro/namespace s)
                             (namespace schema-name)))]
      (->> (assoc-some s
                       :avro/name avro-n
                       :avro/namespace avro-ns)
           (filter-keys #(ns= :avro %))
           (map-keys disqualify-key)))
    {:type schema-name}))

(defn- avro-type [registry schema-name]
  (let [s (get registry schema-name)]
    (keyword (or (:avro/type s)
                 (cond
                   (:scribe/keys s) :record
                   (:scribe/coll-of s) :array)))))

(declare ->avro-schema)

(defmulti ^:private ->complex-avro-schema
  (fn [registry schema-name]
    (avro-type registry schema-name)))

(defmethod ->complex-avro-schema :array [registry schema-name]
  (let [s  (get registry schema-name)
        is (:scribe/coll-of s)]
    {:items (->avro-schema registry is)
     :type  :array}))

(defmethod ->complex-avro-schema :record [registry schema-name]
  (let [s      (get registry schema-name)
        ks     (:scribe/keys s)
        ;; TODO req, opt-un, opt
        req-un (:req-un ks)]
    {:fields (map (partial ->avro-schema registry) req-un)
     :type   :record}))

(defmethod ->complex-avro-schema :default [_ _] nil)

(defn avro-logical-type [registry schema-name]
  (let [s (get registry schema-name)]
    (keyword (or (:avro/logicalType s)
                 (:scribe/type s)))))

(defmulti ^:private ->logical-avro-schema
  (fn [registry schema-name]
    (avro-logical-type registry schema-name)))

(defmethod ->logical-avro-schema :decimal [registry schema-name]
  (let [s (get registry schema-name)]
    {:type        :bytes
     :logicalType :decimal
     :precision   (:scribe/precision s)
     :scale       (:scribe/scale s)}))

(defmethod ->logical-avro-schema :date [registry schema-name]
  (let [s (get registry schema-name)]
    {:type        :int
     :logicalType :date}))

(defmethod ->logical-avro-schema :time-millis [registry schema-name]
  (let [s (get registry schema-name)]
    {:type        :int
     :logicalType :time-millis}))

(defmethod ->logical-avro-schema :time-micros [registry schema-name]
  (let [s (get registry schema-name)]
    {:type        :long
     :logicalType :time-micros}))

(defmethod ->logical-avro-schema :timestamp-millis [registry schema-name]
  (let [s (get registry schema-name)]
    {:type        :int
     :logicalType :timestamp-millis}))

(defmethod ->logical-avro-schema :timestamp-micros [registry schema-name]
  (let [s (get registry schema-name)]
    {:type        :long
     :logicalType :timestamp-micros}))

(defmethod ->logical-avro-schema :default [registry schema-name] nil)

(defn ->avro-schema [registry schema-name]
  (merge (->base-avro-schema registry schema-name)
         (->logical-avro-schema registry schema-name)
         (->complex-avro-schema registry schema-name)))

;;,-----------------------
;;| Avro pre-serialization
;;`-----------------------

(defn pre-serialize [])
