(ns scribe.avro.core
  "Scribe spec to Avro schema conversions, and Avro
  pre-serialization/post-deserialization."
  (:require [medley.core :refer [assoc-some filter-keys map-keys]]
            [scribe.avro.serde-utils :as serde]
            [scribe.avro.specs :as avro.specs]))

(defn- name+ [x] (when x (name x)))

(defn- disqualify-key [k]
  (-> k name keyword))

(defn- ns= [ns k]
  (= (name ns) (namespace k)))

(defn- update-some
  "Updates a key with a fn in a map, if and only if the value in the map is present."
  ([m k f]
   (if-not (find m k) m (update m k f)))
  ([m k f & kfs]
   (reduce (fn [m [k f]] (update-some m k f))
           (update-some m k f)
           (partition 2 kfs))))

;;,------------------------
;;| Avro Schema conversions
;;`------------------------
(defn- ->base-avro-schema [registry spec-name]
  (if-let [s (get registry spec-name)]
    ;; * if there isn't anything, they're not present
    ;; * then, in order of preference:
    ;;   - :avro/type | :avro/name | :avro/namespace
    ;;   - :scribe/type | (name spec-name) | (namespace spec-name)
    ;; * namify if keywords
    (-> {}
        (assoc-some :avro/name (name spec-name)
                    :avro/namespace (namespace spec-name)
                    :avro/type (:scribe/type s))
        (merge s)
        (update-some :avro/name name+
                     :avro/namespace name+)
        (->> (filter-keys #(ns= :avro %))
             (map-keys disqualify-key)))
    {:type spec-name}))

(defn- avro-type [registry spec-name]
  (let [s (get registry spec-name)]
    (keyword (or (:avro/type s)
                 (:scribe/type s)
                 (cond
                   (:scribe/keys s) :record
                   (:scribe/coll-of s) :array)))))

(declare ->avro-schema)

(defmulti ^:private ->complex-avro-schema
  (fn [registry spec-name]
    (avro-type registry spec-name)))

(defmethod ->complex-avro-schema :array [registry spec-name]
  (let [s  (get registry spec-name)
        is (:scribe/coll-of s)]
    {:items (->avro-schema registry is)
     :type  :array}))

(defmethod ->complex-avro-schema :record [registry spec-name]
  (let [s      (get registry spec-name)
        ks     (:scribe/keys s)
        ;; TODO req, opt-un, opt
        req-un (:req-un ks)]
    {:fields (map (partial ->avro-schema registry) req-un)
     :type   :record}))

(defmethod ->complex-avro-schema :default [_ _] nil)

(defn- avro-logical-type [registry spec-name]
  (let [s (get registry spec-name)]
    (keyword (or (:avro/logicalType s)
                 (:scribe/type s)))))

(defmulti ^:private ->logical-avro-schema
  (fn [registry spec-name]
    (avro-logical-type registry spec-name)))

(defmethod ->logical-avro-schema :decimal [registry spec-name]
  (let [s (get registry spec-name)]
    {:type        :bytes
     :logicalType :decimal
     :precision   (:scribe/precision s)
     :scale       (:scribe/scale s)}))

(defmethod ->logical-avro-schema :date [registry spec-name]
  (let [s (get registry spec-name)]
    {:type        :int
     :logicalType :date}))

(defmethod ->logical-avro-schema :time-millis [registry spec-name]
  (let [s (get registry spec-name)]
    {:type        :int
     :logicalType :time-millis}))

(defmethod ->logical-avro-schema :time-micros [registry spec-name]
  (let [s (get registry spec-name)]
    {:type        :long
     :logicalType :time-micros}))

(defmethod ->logical-avro-schema :timestamp-millis [registry spec-name]
  (let [s (get registry spec-name)]
    {:type        :int
     :logicalType :timestamp-millis}))

(defmethod ->logical-avro-schema :timestamp-micros [registry spec-name]
  (let [s (get registry spec-name)]
    {:type        :long
     :logicalType :timestamp-micros}))

(defmethod ->logical-avro-schema :default [registry spec-name] nil)

(defn ->avro-schema
  "Generates an Avro schema for spec-name.

  registry : a map containing Scribe specs, keyed by spec-name
  spec-name: the key of the spec to convert"
  [registry spec-name]
  (merge (->base-avro-schema registry spec-name) ;; TODO pass in a spec here
         (->logical-avro-schema registry spec-name) ;; TODO pass in a spec here
         (->complex-avro-schema registry spec-name))) ;; NOTE this is recursive, needs whole registry

(defn- primitive-type? [spec]
  (avro.specs/primitive-type? (:scribe/type spec)))

(defn- logical-type? [spec]
  (avro.specs/logical-type? (:scribe/type spec)))

;;,-----------------------
;;| Avro pre-serialization
;;`-----------------------
(defmulti ^:private logical-type-pre-serialize
  (fn [spec _] (:scribe/type spec)))

(defmethod logical-type-pre-serialize :decimal [spec value-to-serialize]
  (serde/big-decimal->unscaled-int-bytes (:scribe/scale spec)
                                         ;; TODO ??? soo.. precision? check it in the scribe.spec.core?
                                         value-to-serialize))

(defn pre-serialize

  "Prepares a value for encoding to Avro.

  For example, it converts `BigDecimal`s to bytes for `:decimal`s,
  `:timestamp-millis` to integers and so on, according to the Avro
  specification."

  [registry spec-name value-to-serialize]
  (let [spec (get registry spec-name)]
    (cond
      (primitive-type? spec) value-to-serialize
      (logical-type? spec)   (logical-type-pre-serialize spec value-to-serialize))))

;;,------------------------
;;| Avro post-deserialization
;;`------------------------
(defmulti ^:private logical-type-post-serialize
  (fn [spec _] (:scribe/type spec)))

(defmethod logical-type-post-serialize :decimal [spec deserialized-value]
  (serde/unscaled-int-bytes->big-decimal (:scribe/scale spec)
                                         ;; TODO ??? soo.. precision? check it in the scribe.spec.core?
                                         deserialized-value))

(defn post-deserialize

  "Processes a deserailized value from Avro.

  For example, it converts bytes to a BigDecimal for `:decimal`s, integers to
  Instants for `:timestamp-millis` and so on, according to the Avro
  specification."

  [registry spec-name deserialized-value]
  (let [spec (get registry spec-name)]
    (cond
      (primitive-type? spec) deserialized-value
      (logical-type? spec)   (logical-type-post-serialize spec deserialized-value))))
