(ns scribe.avro.core
  "Scribe spec to Avro schema conversions, and Avro
  pre-serialization/post-deserialization."
  (:require [clojure.spec.alpha :as s]
            [medley.core :refer [assoc-some]]
            [scribe.avro.serde-utils :as serde]
            scribe.specs))

;; TODO refactor ->avro-schema, pre-serialize and post-deserialize into just multimethods

(defn- name+ [x] (when x (name x)))

(defn- disqualify-key [k]
  (-> k name keyword))

(defn- ns= [ns k]
  (= (name ns) (namespace k)))

(defn- update-some
  "Updates a key with a fn in a map, if and only if the value in the map is present."
  ([m k f]
   (if-not (find m k) m (update m k f)))
  ([m k f & kfs]
   (reduce (fn [m [k f]] (update-some m k f))
           (update-some m k f)
           (partition 2 kfs))))

(defn- scribe-spec-variant [registry spec-name]
  (or (scribe.specs/variant (get registry spec-name))
      spec-name))

;;,------------------------
;;| Avro Schema conversions
;;`------------------------
(defn- ->n+ns [spec-name]
  (-> {:name (name spec-name)}
      (assoc-some :namespace (namespace spec-name))))

(s/fdef ->avro-schema
        :args (s/cat :registry (s/map-of keyword? :scribe.specs/spec)
                     :spec-name keyword?))

(defmulti ->avro-schema
  "Generates an Avro schema for spec-name.

  registry : a map containing Scribe specs, keyed by spec-name
  spec-name: the key of the spec to convert"
  scribe-spec-variant)

(defmethod ->avro-schema :decimal [registry spec-name]
  (let [s (get registry spec-name)]
    (merge (->n+ns spec-name)
           {:type        :bytes
            :logicalType :decimal
            :precision   (:scribe/precision s)
            :scale       (:scribe/scale s)})))

(defmethod ->avro-schema :date [registry spec-name]
  (let [s (get registry spec-name)]
    (merge (->n+ns spec-name)
           {:type        :int
            :logicalType :date})))

(defmethod ->avro-schema :time-millis [registry spec-name]
  (let [s (get registry spec-name)]
    (merge (->n+ns spec-name)
           {:type        :int
            :logicalType :time-millis})))

(defmethod ->avro-schema :time-micros [registry spec-name]
  (let [s (get registry spec-name)]
    (merge (->n+ns spec-name)
           {:type        :long
            :logicalType :time-micros})))

(defmethod ->avro-schema :timestamp-millis [registry spec-name]
  (let [s (get registry spec-name)]
    (merge (->n+ns spec-name)
           {:type        :int
            :logicalType :timestamp-millis})))

(defmethod ->avro-schema :timestamp-micros [registry spec-name]
  (let [s (get registry spec-name)]
    (merge (->n+ns spec-name)
           {:type        :long
            :logicalType :timestamp-micros})))

(defmethod ->avro-schema :enum [registry spec-name]
  (let [s  (get registry spec-name)]
    (merge (->n+ns spec-name)
           {:symbols (:scribe/enum s)
            :type    :enum})))

(defmethod ->avro-schema :coll-of [registry spec-name]
  (let [s  (get registry spec-name)
        is (:scribe/coll-of s)]
    (merge (->n+ns spec-name)
           {:items (->avro-schema registry is)
            :type  :array})))

(defmethod ->avro-schema :keys [registry spec-name]
  (let [s      (get registry spec-name)
        ks     (:scribe/keys s)
        ;; TODO req, opt-un, opt
        req-un (:req-un ks)]
    (merge (->n+ns spec-name)
           {:fields (map (partial ->avro-schema registry) req-un)
            :type   :record})))

(defmethod ->avro-schema :or [registry spec-name]
  (let [s (get registry spec-name)]
    (->> (:scribe/or s)
         (partition 2)
         (map (fn [[_k s]]
                (->avro-schema registry s))))))

;; HACK FIXME ??? should just `derive` explicitly primitive types from a ::primitive
;; TODO :default should throw (empty impl) ???
(defmethod ->avro-schema :default [registry spec-name]
  (if-let [s-raw (get registry spec-name)]
    ;; TODO s/conform into :scribe/type here, get rid of this shit
    (let [s (if (associative? s-raw)
              s-raw
              {:scribe/type (-> s-raw name keyword)})]
      (-> (->n+ns spec-name)
          (assoc :type (:scribe/type s))))
    {:type spec-name}))

;;,-----------------------
;;| Avro pre-serialization
;;`-----------------------
(declare pre-serialize)

(defmulti ^:private pre-serialize-
  (fn [registry spec-name _] (scribe-spec-variant registry spec-name)))

(defmethod pre-serialize- :decimal [registry spec-name value-to-serialize]
  (let [s (get registry spec-name)]
    (serde/big-decimal->unscaled-int-bytes (:scribe/scale s)
                                           ;; TODO ??? soo.. precision? check it in the scribe.spec.core?
                                           value-to-serialize)))

(defmethod pre-serialize- :coll-of [registry spec-name value-to-deserialize]
  (let [spec      (get registry spec-name)
        item-spec (:scribe/coll-of spec)]
    (map (partial pre-serialize registry item-spec) value-to-deserialize)))

(defmethod pre-serialize- :keys [registry spec-name deserialized-value]
  (let [spec        (get registry spec-name)
        keys-ks     (:scribe/keys spec)
        req-un-keys (:req-un keys-ks)]
    ;; TODO req, opt, opt-un
    (reduce (fn [m k-spec-name]
              (update m
                      (disqualify-key k-spec-name)
                      (partial pre-serialize registry k-spec-name)))
            deserialized-value
            req-un-keys)))

;; HACK FIXME ??? should just `derive` explicitly primitive types from a ::primitive
;; TODO :default should throw (empty impl) ???
(defmethod pre-serialize- :default [_ _ value-to-serialize] value-to-serialize)

(defn pre-serialize
  "Prepares a value for encoding to Avro.

  For example, it converts `BigDecimal`s to bytes for `:decimal`s,
  `:timestamp-millis` to integers and so on, according to the Avro
  specification."
  [registry spec-name value-to-serialize]
  (pre-serialize- registry spec-name value-to-serialize))

;;,------------------------
;;| Avro post-deserialization
;;`------------------------
(declare post-deserialize)

(defmulti ^:private post-deserialize-
  (fn [registry spec-name _] (scribe-spec-variant registry spec-name)))

(defmethod post-deserialize- :decimal [registry spec-name deserialized-value]
  (let [spec (get registry spec-name)]
    (serde/unscaled-int-bytes->big-decimal (:scribe/scale spec)
                                           ;; TODO ??? soo.. precision? check it in the scribe.spec.core?
                                           deserialized-value)))

(defmethod post-deserialize- :coll-of [registry spec-name deserialized-value]
  (let [spec      (get registry spec-name)
        item-spec (:scribe/coll-of spec)]
    (map (partial post-deserialize registry item-spec) deserialized-value)))

(defmethod post-deserialize- :keys [registry spec-name deserialized-value]
  (let [spec        (get registry spec-name)
        keys-ks     (:scribe/keys spec)
        req-un-keys (:req-un keys-ks)]
    ;; TODO req, opt, opt-un
    (reduce (fn [m k-spec-name]
              (update m
                      (disqualify-key k-spec-name)
                      (partial post-deserialize registry k-spec-name)))
            deserialized-value
            req-un-keys)))

;; HACK FIXME ??? should just `derive` explicitly primitive types from a ::primitive
;; TODO :default should throw (empty impl) ???
(defmethod post-deserialize- :default [_ _ deserialized-value] deserialized-value)

(defn post-deserialize
  "Processes a deserailized value from Avro.

  For example, it converts bytes to a BigDecimal for `:decimal`s, integers to
  Instants for `:timestamp-millis` and so on, according to the Avro
  specification."
  [registry spec-name deserialized-value]
  (post-deserialize- registry spec-name deserialized-value))
