(ns scribe.avro.core
  "Scribe spec to Avro schema conversions, and Avro
  pre-serialization/post-deserialization."
  (:require [clojure.spec.alpha :as s]
            [medley.core :refer [assoc-some map-keys]]
            [scribe.avro.serde-utils :as serde]
            scribe.specs))

;;,-----------------------------------------------
;;| https://avro.apache.org/docs/current/spec.html
;;`-----------------------------------------------

;; TODO refactor ->avro-schema, pre-serialize and post-deserialize into just multimethods

(defn- disqualify-key [k] (-> k name keyword))
(defn- qualify-key [ns k] (keyword (name ns) (name k)))

(defn- ns= [ns k] (= (name ns) (namespace k)))

(defn- update-some
  "Updates a key with a fn in a map, if and only if the value in the map is
  present."
  ([m k f]
   (if-not (find m k) m (update m k f)))
  ([m k f & kfs]
   (reduce (fn [m [k f]] (update-some m k f))
           (update-some m k f)
           (partition 2 kfs))))

(defn- ->avro-n+ns [spec-name]
  (-> {:name (name spec-name)}
      (assoc-some :namespace (namespace spec-name))))

(defn- ->avro-ref [avro-n+ns] (map-keys (partial qualify-key :avro) avro-n+ns))

(defn- assoc-avro-ref
  "To be called after defining an avro type so that subsequent calls will just
  use its fully qualified name"
  [registry spec-name]
  (assoc registry spec-name (-> spec-name
                                ->avro-n+ns
                                ->avro-ref)))

(s/def :scribe.avro/ref
  (s/keys :req [:avro/name]
          :opt [:avro/namespace]))

(defn- spec-variant [registry spec-name]
  (let [s (get registry spec-name)]
    (or (scribe.specs/variant s)
        (when (s/valid? :scribe.avro/ref s)
          :scribe.avro/ref)
        spec-name)))

;;,------------------------
;;| Avro Schema conversions
;;`------------------------
(defmulti ^:private ->avro-schema-
  "Returns a tuple: registry+schema. The registry is updated so that schemas are
  only defined on the first instance and then referred to by fully qualified
  name."
  spec-variant)

(defmethod ->avro-schema- :scribe.avro/ref [registry spec-name]
  (let [s  (get registry spec-name)
        n  (:avro/name s)
        ns (:avro/namespace s)]
    [registry (if ns (str ns "." n) n)]))

(defmethod ->avro-schema- :decimal [registry spec-name]
  (let [s (get registry spec-name)]
    [registry
     {:type        :bytes
      :logicalType :decimal
      :precision   (:scribe/precision s)
      :scale       (:scribe/scale s)}]))

(defmethod ->avro-schema- :date [registry spec-name]
  (let [s (get registry spec-name)]
    [registry
     {:type        :int
      :logicalType :date}]))

(defmethod ->avro-schema- :time-millis [registry spec-name]
  (let [s (get registry spec-name)]
    [registry
     {:type        :int
      :logicalType :time-millis}]))

(defmethod ->avro-schema- :time-micros [registry spec-name]
  (let [s (get registry spec-name)]
    [registry
     {:type        :long
      :logicalType :time-micros}]))

(defmethod ->avro-schema- :timestamp-millis [registry spec-name]
  (let [s (get registry spec-name)]
    [registry
     {:type        :int
      :logicalType :timestamp-millis}]))

(defmethod ->avro-schema- :timestamp-micros [registry spec-name]
  (let [s (get registry spec-name)]
    [registry
     {:type        :long
      :logicalType :timestamp-micros}]))

(defmethod ->avro-schema- :enum [registry spec-name]
  (let [s (get registry spec-name)]
    [(assoc-avro-ref registry spec-name)
     (merge (->avro-n+ns spec-name)
            {:symbols (:scribe/enum s)
             :type    :enum})]))

(defmethod ->avro-schema- :coll-of [registry spec-name]
  (let [s              (get registry spec-name)
        is             (:scribe/coll-of s)
        [registry
         items-schema] (->avro-schema- registry is)
        schema         (merge (->avro-n+ns spec-name)
                              {:items items-schema
                               :type  :array})]
    [registry schema]))

(defn- ->record-field [registry spec-name]
  (let [[registry schema] (->avro-schema- registry spec-name)]
    [registry {:name (name spec-name)
               :type schema}]))

(defmethod ->avro-schema- :keys [registry spec-name]
  (let [s      (get registry spec-name)
        ks     (:scribe/keys s)
        ;; TODO req, opt-un, opt
        req-un (:req-un ks)
        [registry fields]
        (reduce (fn [[registry field-schemas] spec-name]
                  (let [[registry field-schema] (->record-field registry spec-name)]
                    [registry (conj field-schemas field-schema)]))
                [registry []]
                req-un)]
    [(assoc-avro-ref registry spec-name)
     (merge (->avro-n+ns spec-name)
            {:fields fields
             :type   :record})]))

(defmethod ->avro-schema- :or [registry spec-name]
  (let [s (get registry spec-name)]
    (->> (:scribe/or s)
         (reduce (fn [[registry members-schemas] spec-name]
                   (let [[registry member-schema] (->avro-schema- registry spec-name)]
                     [registry (conj members-schemas member-schema)]))
                 [registry []]))))

;; HACK FIXME ??? should just `derive` explicitly primitive types from a ::primitive
;; TODO :default should throw (empty impl) ???
(defmethod ->avro-schema- :default [registry spec-name]
  [registry
   (let [s-raw     (get registry spec-name)
         ;; TODO s/conform into :scribe/type here, get rid of this shit
         avro-type (cond
                     (associative? s-raw) (:scribe/type s-raw)
                     (nil? s-raw)         spec-name
                     :else                (-> s-raw name keyword))]
     {:type avro-type})])

(s/fdef ->avro-schema
        :args (s/cat :registry (s/map-of keyword? :scribe.specs/spec)
                     :spec-name keyword?))
(defn ->avro-schema
  "Generates an Avro schema for spec-name.

  registry : a map containing Scribe specs, keyed by spec-name
  spec-name: the key of the spec to convert"
  [registry spec-name]
  (second (->avro-schema- registry spec-name)))

;;,-----------------------
;;| Avro pre-serialization
;;`-----------------------
(declare pre-serialize)

(defmulti ^:private pre-serialize-
  (fn [registry spec-name _] (spec-variant registry spec-name)))

(defmethod pre-serialize- :decimal [registry spec-name value-to-serialize]
  (let [s (get registry spec-name)]
    (serde/big-decimal->unscaled-int-bytes (:scribe/scale s)
                                           ;; TODO ??? soo.. precision? check it in the scribe.spec.core?
                                           value-to-serialize)))

(defmethod pre-serialize- :coll-of [registry spec-name value-to-serialize]
  (let [spec      (get registry spec-name)
        item-spec (:scribe/coll-of spec)]
    (map (partial pre-serialize registry item-spec) value-to-serialize)))

(defmethod pre-serialize- :keys [registry spec-name value-to-serialize]
  (let [spec        (get registry spec-name)
        keys-ks     (:scribe/keys spec)
        req-un-keys (:req-un keys-ks)]
    ;; TODO req, opt, opt-un
    (reduce (fn [m k-spec-name]
              (update m
                      (disqualify-key k-spec-name)
                      (partial pre-serialize registry k-spec-name)))
            value-to-serialize
            req-un-keys)))

;; HACK FIXME ??? should just `derive` explicitly primitive types from a ::primitive
;; TODO :default should throw (empty impl) ???
(defmethod pre-serialize- :default [_ _ value-to-serialize] value-to-serialize)

(defn pre-serialize
  "Prepares a value for encoding to Avro.

  For example, it converts `BigDecimal`s to bytes for `:decimal`s,
  `:timestamp-millis` to integers and so on, according to the Avro
  specification."
  [registry spec-name value-to-serialize]
  (pre-serialize- registry spec-name value-to-serialize))

;;,------------------------
;;| Avro post-deserialization
;;`------------------------
(declare post-deserialize)

(defmulti ^:private post-deserialize-
  (fn [registry spec-name _] (spec-variant registry spec-name)))

(defmethod post-deserialize- :decimal [registry spec-name deserialized-value]
  (let [spec (get registry spec-name)]
    (serde/unscaled-int-bytes->big-decimal (:scribe/scale spec)
                                           ;; TODO ??? soo.. precision? check it in the scribe.spec.core?
                                           deserialized-value)))

(defmethod post-deserialize- :coll-of [registry spec-name deserialized-value]
  (let [spec      (get registry spec-name)
        item-spec (:scribe/coll-of spec)]
    (map (partial post-deserialize registry item-spec) deserialized-value)))

(defmethod post-deserialize- :keys [registry spec-name deserialized-value]
  (let [spec        (get registry spec-name)
        keys-ks     (:scribe/keys spec)
        req-un-keys (:req-un keys-ks)]
    ;; TODO req, opt, opt-un
    (reduce (fn [m k-spec-name]
              (update m
                      (disqualify-key k-spec-name)
                      (partial post-deserialize registry k-spec-name)))
            deserialized-value
            req-un-keys)))

;; HACK FIXME ??? should just `derive` explicitly primitive types from a ::primitive
;; TODO :default should throw (empty impl) ???
(defmethod post-deserialize- :default [_ _ deserialized-value] deserialized-value)

(defn post-deserialize
  "Processes a deserailized value from Avro.

  For example, it converts bytes to a BigDecimal for `:decimal`s, integers to
  Instants for `:timestamp-millis` and so on, according to the Avro
  specification."
  [registry spec-name deserialized-value]
  (post-deserialize- registry spec-name deserialized-value))
