(ns scribe.specs
  "Clojure spec definitions for scribe specs."
  (:require [clojure.spec.alpha :as s]))

(s/def :scribe/type keyword?)

(defn- type= [t m]
  (= t (:scribe/type m)))

(defn- ->type [type]
  (s/and (s/keys :req [:scribe/type])
         (partial type= type)))

(s/def ::string
  (s/or :wrapped (->type :string)
        :naked #(= :scribe/string %)))

(s/def ::boolean
  (s/or :wrapped (->type :boolean)
        :naked #(= :scribe/boolean %)))

(s/def ::nil
  (s/or :wrapped (->type :nil)
        :naked #(= :scribe/nil %)))

;; TODO ::short (avro "int")
;; TODO ::long (avro "long")
;; TODO ::float (avro "double")
(s/def ::double
  (s/or :wrapped (->type :double)
        :naked #(= :scribe/double %)))

(s/def :scribe/precision integer?)
(s/def :scribe/scale integer?)
(s/def ::decimal
  (s/and (->type :decimal)
         (s/keys :req [:scribe/precision
                       :scribe/scale])))

(s/def ::date (->type :date))
(s/def ::timestamp-micros (->type :timestamp-micros))
(s/def ::timestamp-millis (->type :timestamp-millis))
(s/def ::time-millis (->type :time-millis))
(s/def ::time-micros (->type :time-micros))

(s/def :scribe/enum set?)
(s/def ::enum (s/keys :req [:scribe/enum]))

(s/def :scribe/coll-of
  (s/or :spec-ref keyword?
        :scribe/spec :scribe/spec))
(s/def ::coll-of
  (s/keys :req [:scribe/coll-of]))

(s/def ::keys-coll (s/coll-of keyword?))
(s/def :scribe.keys/req ::keys-coll)
(s/def :scribe.keys/req-un ::keys-coll)
(s/def :scribe.keys/opt ::keys-coll)
(s/def :scribe.keys/opt-un ::keys-coll)
(s/def :scribe/keys
  (s/and (s/keys ::opt #{:scribe.keys/req
                         :scribe.keys/req-un
                         :scribe.keys/opt
                         :scribe.keys/opt-un})
         #(some #{:req :req-un :opt :opt-un} (keys %))))
(s/def ::keys
  (s/and (s/keys :req [:scribe/keys])))

(s/def :scribe/or (s/coll-of keyword?))
(s/def ::or (s/keys :req [:scribe/or]))

(s/def :scribe/spec
  (s/or :scribe/string ::string
        :scribe/boolean ::boolean
        :scribe/nil ::nil
        :scribe/double ::double
        ;;;
        :scribe/decimal ::decimal
        :scribe/date ::date
        :scribe/timestamp-millis ::timestamp-millis
        :scribe/timestamp-micros ::timestamp-micros
        :scribe/time-millis ::time-millis
        :scribe/time-micros ::time-micros
        ;;;
        :scribe/enum ::enum
        :scribe/coll-of ::coll-of
        :scribe/keys ::keys
        :scribe/or ::or))
(s/def ::spec :scribe/spec)

(defn variant
  "Returns the variant of the given scribe spec

  (e.g. :string, :decimal, :time-millis, :coll-of, etc.)"
  [x]
  (when (s/valid? ::spec x)
    (first (s/conform ::spec x))))
