(ns scribe.avro.core
  "Scribe spec to Avro schema conversions, and Avro
  pre-serialization/post-deserialization."
  (:require [abracad.avro :as avro]
            [clojure.spec.alpha :as s]
            [medley.core :refer [assoc-some]]
            [scribe.avro.serde-utils :as serde]
            [scribe.utils :refer :all]))

(def avro-decimal-precision 32)
(def avro-decimal-scale 32)

;;,-----------------------------------------------
;;| https://avro.apache.org/docs/current/spec.html
;;`-----------------------------------------------

(defn- avro-enum?
  "non-empty, keyword-only sets are Avro enums"
  [x]
  ((every-pred set?
               not-empty
               (partial every? keyword?)) x))

(defn- spec-variant [spec]
  (cond
    (symbol? spec) spec
    :else
    (let [form (s/form spec)
          _    (type form)]
      (cond
        (avro-enum? form) :scribe.avro/enum
        (symbol? form)    form
        (seq? form)       (first form)))))

(defn- ->avro-n+ns [spec]
  (-> {:name (name spec)}
      (assoc-some :namespace (namespace spec))))

(defn- ->avro-ref [spec]
  (-> (->avro-n+ns spec)
      (qualify-keys :avro)
      (with-meta {:spec-variant (spec-variant spec)})))

(defn- assoc-avro-ref
  "To be called after defining an avro type so that subsequent calls will just
  use its fully qualified name"
  [avro-refs spec]
  (assoc avro-refs spec (->avro-ref spec)))

;;,------------------------
;;| Avro Schema conversions
;;`------------------------
(defmulti ^:private ->avro-schema-
  "Returns a tuple: avro-refs+schema. The avro-refs is updated so that schemas are
  only defined on the first instance and then referred to by fully qualified
  name."
  (fn [avro-refs spec]
    (if (get avro-refs spec)
      :scribe.avro/ref
      (spec-variant spec))))

(defmethod ->avro-schema- :scribe.avro/ref
  [avro-refs spec]
  (let [s  (get avro-refs spec)
        n  (:avro/name s)
        ns (:avro/namespace s)]
    [avro-refs (if ns (str ns "." n) n)]))

(defmethod ->avro-schema- 'clojure.core/string?
  [avro-refs _]
  [avro-refs {:type :string}])

(defmethod ->avro-schema- 'clojure.core/boolean?
  [avro-refs _]
  [avro-refs {:type :boolean}])

(defmethod ->avro-schema- 'clojure.core/decimal?
  [avro-refs spec]
  ;; TODO ??? how do we treat custom precision/scale? dynamics? some special
  ;; specs? either way it's gonna be yucky
  [avro-refs {:type :bytes
              :logicalType    :decimal
              :precision      avro-decimal-precision
              :scale          avro-decimal-scale}])

(defmethod ->avro-schema- 'scribe.time-specs/instant?
  [avro-refs spec]
  ;; HACK TODO it's gonna be millis for now because that's what we need..
  #_{:type        :int
     :logicalType :timestamp-micros}
  {:type        :int
   :logicalType :timestamp-millis})

(defmethod ->avro-schema- 'scribe.time-specs/local-date?
  [avro-refs spec]
  (let [s (get avro-refs spec)]
    [avro-refs
     {:type        :int
      :logicalType :date}]))

(defmethod ->avro-schema- 'scribe.time-specs/local-time?
  [avro-refs spec]
  ;; HACK TODO it's gonna be millis for now because that's what we need..
  #_{:type        :long
     :logicalType :time-micros}
  {:type        :int
   :logicalType :time-millis})

(defmethod ->avro-schema- 'scribe.time-specs/period?
  [avro-refs spec]
  (merge (->avro-n+ns spec)
         {:type        :fixed
          :size        12
          :logicalType :duration}))

(defmethod ->avro-schema- :scribe.avro/enum
  [avro-refs spec]
  [(assoc-avro-ref avro-refs spec)
   (merge (->avro-n+ns spec)
          {:symbols (s/form spec)
           :type    :enum})])

(defmethod ->avro-schema- 'clojure.spec.alpha/coll-of
  [avro-refs spec]
  (let [form (s/form spec)
        ;; TODO support opts ???
        pred (second form)
        [avro-refs
         items-schema] (->avro-schema- avro-refs pred)
        schema         (merge (->avro-n+ns spec)
                              {:items items-schema
                               :type  :array})]
    [avro-refs schema]))

(defn- ->record-field
  [avro-refs spec]
  (let [[avro-refs schema] (->avro-schema- avro-refs spec)]
    [avro-refs {:name (name spec)
                :type schema}]))
(defmethod ->avro-schema- 'clojure.spec.alpha/keys
  [avro-refs spec]
  (let [form   (s/form spec)
        opts   (kwargs->m (rest form))
        ;; TODO req, opt-un, opt
        req-un (:req-un opts)
        [avro-refs fields]
        (reduce (fn [[avro-refs field-schemas] spec]
                  (let [[avro-refs field-schema] (->record-field avro-refs spec)]
                    [avro-refs (conj field-schemas field-schema)]))
                [avro-refs []]
                req-un)]
    [(assoc-avro-ref avro-refs spec)
     (merge (->avro-n+ns spec)
            {:fields fields
             :type   :record})]))

(defn- or-preds
  [spec]
  (-> spec s/form rest kwargs->m vals))
(defmethod ->avro-schema- 'clojure.spec.alpha/or
  [avro-refs spec]
  (let [form (s/form spec)
        preds (-> (rest form) kwargs->m vals)]
    (->> preds
         (reduce (fn [[avro-refs members-schemas] spec]
                   (let [[avro-refs member-schema]
                         (->avro-schema- avro-refs spec)]
                     [avro-refs (conj members-schemas member-schema)]))
                 [avro-refs []]))))

(defmacro try-or-nil [& body]
  `(try
     ~@body
     (catch Exception _# nil)))
(defn merge-avro-schemas [s1 s2]
  ;; TODO merge keys
  (merge s1 s2))
(defmethod ->avro-schema- 'clojure.spec.alpha/and [avro-refs spec]
  (let [form  (s/form spec)
        preds (rest form)]
    (->> preds
         (reduce (fn [[avro-refs avro-schema] p]
                   (let [[a-rfs a-s] (try-or-nil (->avro-schema- avro-refs p))]
                     (if a-s
                       [a-rfs (merge-avro-schemas avro-schema a-s)]
                       [avro-refs avro-schema])))
                 [avro-refs nil]))))

(defmethod ->avro-schema- :default
  [avro-refs spec]
  (throw (ex-info (str "could not determine Avro schema for" spec))))

(defn ->avro-schema
  "Generates an Avro schema for spec-k.

  spec-k: the key of the spec to convert"
  [spec-k]
  (second (->avro-schema- {} spec-k)))

;;,-----------------------
;;| Avro pre-serialization
;;`-----------------------
(declare pre-serialize)

(defmulti ^:private pre-serialize- (fn [spec _v] (spec-variant spec)))

(defmethod pre-serialize- 'clojure.core/decimal?
  [spec v]
  ;; TODO ??? soo.. precision? check it in the scribe.spec.core?
  (serde/big-decimal->unscaled-int-bytes avro-decimal-scale
                                         v))

(defmethod pre-serialize- 'clojure.spec.alpha/coll-of
  [spec v]
  (let [pred (-> spec s/form second)]
    (map (partial pre-serialize pred) v)))

(defmethod pre-serialize- 'clojure.spec.alpha/keys
  [spec v]
  (let [form        (s/form spec)
        opts        (kwargs->m (rest form))
        req-un-keys (:req-un opts)]
    ;; TODO req, opt, opt-un
    ;; TODO simplify with map-kv
    (reduce (fn [m k-spec]
              (update m
                      (disqualify-key k-spec)
                      (partial pre-serialize k-spec)))
            v
            req-un-keys)))

(defn- or-pred-for [or-spec v]
  "Given an s/or spec, and a value v, it returns the pred for v"
  (let [form       (s/form or-spec)
        matching-k (first (s/conform or-spec v))
        k->pred    (kwargs->m (rest form))]
    (k->pred matching-k)))
(defmethod pre-serialize- 'clojure.spec.alpha/or
  [spec v]
  (pre-serialize (or-pred-for spec v) v))

(defmethod pre-serialize- 'clojure.spec.alpha/and [spec v]
  (let [preds (seq (rest (s/form spec)))]
    ;; if (empty? preds), return v
    ;; if all preds fail, return v
    ;; otherwise, return first successful (pre-serialize pred v)
    (letfn [(go [p & ps]
                (if-not ps
                  v
                  (try
                    (pre-serialize p v)
                    (catch Exception _
                      (go ps v)))))]
      (apply go preds))))

(defmethod pre-serialize- :default [_ v] v)

(defn pre-serialize
  "Prepares a value for encoding to Avro.

  For example, it converts `BigDecimal`s to bytes for `:decimal`s,
  `:timestamp-millis` to integers and so on, according to the Avro
  specification."
  [spec-k v]
  (pre-serialize- spec-k v))

;;,------------------------
;;| Avro post-deserialization
;;`------------------------
(declare post-deserialize)

(defmulti ^:private post-deserialize- (fn [spec _v] (spec-variant spec)))

(defmethod post-deserialize- 'clojure.core/decimal?
  [spec v]
  ;; TODO ??? soo.. precision? check it in the scribe.spec.core?
  (serde/unscaled-int-bytes->big-decimal avro-decimal-scale v))

(defmethod post-deserialize- :scribe.avro/enum
  [_ v]
  (keyword v))

(defmethod post-deserialize- 'clojure.spec.alpha/keys
  [spec v]
  (let [form        (s/form spec)
        opts        (kwargs->m (rest form))
        req-un-keys (:req-un opts)]
    ;; TODO req, opt, opt-un
    ;; TODO simplify with map-kv
    (reduce (fn [m k-spec]
              (update m
                      (disqualify-key k-spec)
                      (partial post-deserialize k-spec)))
            v
            req-un-keys)))

(defmethod post-deserialize- 'clojure.spec.alpha/coll-of
  [spec v]
  (let [pred (-> spec s/form second)]
    (map (partial post-deserialize pred) v)))

(defn- or-avro-value-spec
  "Given the preds from an s/or form, it returns the spec of the value"
  [or-preds v]
  (let [v-type (-> v type keyword)]
    (->> or-preds
         (filter (partial = v-type))
         first)))
(defmethod post-deserialize- 'clojure.spec.alpha/or
  [spec v]
  (if-let [sn (or-avro-value-spec (or-preds spec) v)]
    (post-deserialize sn v)
    v))

(defmethod post-deserialize- 'clojure.spec.alpha/and [spec v]
  (let [preds (seq (rest (s/form spec)))]
    ;; if (empty? preds), return v
    ;; if all preds fail, return v
    ;; otherwise, return first successful (post-deserialize pred v)
    (letfn [(go [p & ps]
                (if-not ps
                  v
                  (try
                    (post-deserialize p v)
                    (catch Exception _
                      (go ps v)))))]
      (apply go preds))))

(defmethod post-deserialize- :default [_ v] v)

(defn post-deserialize
  "Processes a deserailized value from Avro.

  For example, it converts bytes to a BigDecimal for `:decimal`s, integers to
  Instants for `:timestamp-millis` and so on, according to the Avro
  specification."
  [spec-k v]
  (post-deserialize- spec-k v))

;;,------
;;| SerDe
;;`------

(defn binary-encoded [spec-k & records]
  (let [s (->avro-schema spec-k)
        rs (map (partial pre-serialize spec-k) records)]
    (apply avro/binary-encoded s rs)))

(defn decode [spec-k source]
  (let [s (->avro-schema spec-k)]
    (post-deserialize spec-k (avro/decode s source))))

;; TODO all all the abracads
