(ns scribe.avro.core
  "Scribe spec to Avro schema conversions, and Avro
  pre-serialization/post-deserialization."
  (:require [abracad.avro :as aa]
            [clojure.spec.alpha :as s]
            [clojure.tools.logging :as log]
            [medley.core :refer [assoc-some]]
            [scribe.decimal :as decimal]
            [scribe.utils :refer :all]))

(s/def ::spec
  (s/or :spec-k qualified-keyword?
        :pred symbol?
        :form seq?))
(s/def ::avro-ref (s/keys :req [:avro/name :avro/namespace]))
(s/def ::spec->avro-ref (s/map-of ::spec ::avro-ref))

;;,-----------------------------------------------
;;| https://avro.apache.org/docs/current/spec.html
;;`-----------------------------------------------

(defn- avro-enum?
  "non-empty, keyword-only sets are Avro enums"
  [x]
  ((every-pred set?
               not-empty
               (partial every? keyword?)) x))

(defn- spec-variant [spec]
  ;; FIXME .... idk i think this is starting to be a bit much
  (cond
    (symbol? spec)            spec
    (seq? spec)               (first spec)
    (qualified-keyword? spec) (let [form (s/form spec)
                                    _    (type form)]
                                (cond
                                  (avro-enum? form)           :scribe.avro/enum
                                  (symbol? form)              form
                                  (seq? form)                 (first form)))))

(defn- ->avro-n+ns [spec]
  (-> {:name (name spec)}
      (assoc-some :namespace (namespace spec))))

(defn ->avro-ref [spec]
  (-> (->avro-n+ns spec)
      (qualify-keys :avro)
      (with-meta {:spec-variant (spec-variant spec)})))

(s/fdef assoc-avro-ref
        :args (s/cat :spec->avro-ref ::spec->avro-ref
                     :spec ::spec))
(defn assoc-avro-ref
  "Assocs an ::avro-ref computed form spec to spec->avro-ref.

  To be called after defining an avro type so that subsequent calls will just
  use its fully qualified name."
  [spec->avro-ref spec]
  (assoc spec->avro-ref spec (->avro-ref spec)))

(defn- or-preds [spec] (-> spec s/form rest kwargs->m vals))

;;,------------------------
;;| Avro Schema conversions
;;`------------------------
(s/fdef ->avro-schema*
        :args (s/cat :spec->avro-ref ::spec->avro-ref
                     :spec ::spec))
(defmulti ->avro-schema*
  "Returns a tuple: [spec->avro-ref schema]. The spec->avro-ref map is updated
  so that schemas are only defined on the first instance and then referred to by
  fully qualified name.

  NOTE: you shouldn't  need to call this directly, but you can extend this multi to implement
  schema generation for specs that aren't included."
  (fn [spec->avro-ref spec]
    (if (get spec->avro-ref spec)
      :scribe.avro/ref
      (spec-variant spec))))

(defmethod ->avro-schema* :scribe.avro/ref
  [spec->avro-ref spec]
  (let [s  (get spec->avro-ref spec)
        n  (:avro/name s)
        ns (:avro/namespace s)]
    [spec->avro-ref (if ns (str ns "." n) n)]))

(defmethod ->avro-schema* 'clojure.core/string?
  [spec->avro-ref _]
  [spec->avro-ref {:type :string}])

(defmethod ->avro-schema* 'clojure.core/boolean?
  [spec->avro-ref _]
  [spec->avro-ref {:type :boolean}])

(defmethod ->avro-schema* 'scribe.decimal/->decimal-conformer
  [spec->avro-ref spec]
  (let [{:keys [precision scale]} (kwargs->m (rest spec))]
    [spec->avro-ref {:type        :bytes
                     :logicalType :decimal
                     :precision   precision
                     :scale       scale}]))

(defmethod ->avro-schema* 'scribe.time-specs/instant?
  [spec->avro-ref spec]
  ;; HACK TODO it's gonna be millis for now because that's what we need..
  #_{:type        :int
     :logicalType :timestamp-micros}
  {:type        :int
   :logicalType :timestamp-millis})

(defmethod ->avro-schema* 'scribe.time-specs/local-date?
  [spec->avro-ref spec]
  (let [s (get spec->avro-ref spec)]
    [spec->avro-ref
     {:type        :int
      :logicalType :date}]))

(defmethod ->avro-schema* 'scribe.time-specs/local-time?
  [spec->avro-ref spec]
  ;; HACK TODO it's gonna be millis for now because that's what we need..
  #_{:type        :long
     :logicalType :time-micros}
  {:type        :int
   :logicalType :time-millis})

(defmethod ->avro-schema* 'scribe.time-specs/period?
  [spec->avro-ref spec]
  (merge (->avro-n+ns spec)
         {:type        :fixed
          :size        12
          :logicalType :duration}))

(defmethod ->avro-schema* :scribe.avro/enum
  [spec->avro-ref spec]
  [(assoc-avro-ref spec->avro-ref spec)
   (merge (->avro-n+ns spec)
          {:symbols (s/form spec)
           :type    :enum})])

(defmethod ->avro-schema* 'clojure.spec.alpha/coll-of
  [spec->avro-ref spec]
  (let [form (s/form spec)
        ;; TODO support opts ???
        pred (second form)
        [spec->avro-ref
         items-schema] (->avro-schema* spec->avro-ref pred)
        schema         (merge (->avro-n+ns spec)
                              {:items items-schema
                               :type  :array})]
    [spec->avro-ref schema]))

(defn- ->record-field
  [spec->avro-ref spec]
  (let [[spec->avro-ref schema] (->avro-schema* spec->avro-ref spec)]
    [spec->avro-ref {:name (name spec)
                     :type schema}]))
(defmethod ->avro-schema* 'clojure.spec.alpha/keys
  [spec->avro-ref spec]
  (let [form   (s/form spec)
        opts   (kwargs->m (rest form))
        ;; TODO req, opt-un, opt
        req-un (:req-un opts)
        [spec->avro-ref fields]
        (reduce (fn [[spec->avro-ref field-schemas] spec]
                  (let [[spec->avro-ref field-schema] (->record-field spec->avro-ref spec)]
                    [spec->avro-ref (conj field-schemas field-schema)]))
                [spec->avro-ref []]
                req-un)]
    [(assoc-avro-ref spec->avro-ref spec)
     (merge (->avro-n+ns spec)
            {:fields fields
             :type   :record})]))

(defmethod ->avro-schema* 'clojure.spec.alpha/or
  [spec->avro-ref spec]
  (let [form  (s/form spec)
        preds (->> (rest form) (partition 2) (map second))]
    (->> preds
         (reduce (fn [[spec->avro-ref members-schemas] spec]
                   (let [[spec->avro-ref member-schema]
                         (->avro-schema* spec->avro-ref spec)]
                     [spec->avro-ref (conj members-schemas member-schema)]))
                 [spec->avro-ref []]))))

(defn- merge-avro-schemas [s1 s2]
  ;; TODO merge keys
  (merge s1 s2))
(defmethod ->avro-schema* 'clojure.spec.alpha/and [spec->avro-ref spec]
  (let [form  (s/form spec)
        preds (rest form)]
    (->> preds
         (reduce (fn [[spec->avro-ref avro-schema] p]
                   (let [[a-rfs a-s]
                         (try
                           (->avro-schema* spec->avro-ref p)
                           (catch Exception ex
                             (log/trace ex "catching ex in ->avro-schema, returning nil"
                                        (merge (ex-data ex)
                                               {:p                  p
                                                :avro-schema-so-far avro-schema
                                                :spec->avro-ref     spec->avro-ref
                                                :spec               spec
                                                :form               form
                                                :preds              preds}))
                             nil))]
                     (if a-s
                       [a-rfs (merge-avro-schemas avro-schema a-s)]
                       [spec->avro-ref avro-schema])))
                 [spec->avro-ref nil]))))

(defmethod ->avro-schema* :default
  [spec->avro-ref spec]
  (throw (ex-info "could not determine Avro schema" {:spec spec
                                                     :spec->avro-ref spec->avro-ref})))

(defn ->avro-schema
  "Generates an Avro schema for spec-k.

  spec-k: the key of the spec to convert"
  [spec-k]
  (second (->avro-schema* {} spec-k)))

;;,-----------------------
;;| Avro pre-serialization
;;`-----------------------
(declare pre-serialize)
(s/fdef pre-serialize*
        :args (s/cat :spec ::spec
                     :value-to-serialize any?))
(defmulti pre-serialize*
  "See pre-serialize.

  NOTE: you shouldn't  need to call this directly, but you can extend this multi to implement
  value pre-serialization for specs that aren't included."
  (fn [spec _v] (spec-variant spec)))

(defmethod pre-serialize* 'scribe.decimal/->decimal-conformer
  [spec v]
  (let [{:keys [precision scale]} (kwargs->m (rest spec))]
    (decimal/decimal->bytes v precision scale)))

(defmethod pre-serialize* 'clojure.spec.alpha/coll-of
  [spec v]
  (let [pred (-> spec s/form second)]
    (map (partial pre-serialize pred) v)))

(defmethod pre-serialize* 'clojure.spec.alpha/keys
  [spec v]
  (let [form        (s/form spec)
        opts        (kwargs->m (rest form))
        req-un-keys (:req-un opts)]
    ;; TODO req, opt, opt-un
    ;; TODO simplify with map-kv
    (reduce (fn [m k-spec]
              (update m
                      (disqualify-keyword k-spec)
                      (partial pre-serialize k-spec)))
            v
            req-un-keys)))

(defn- or-pred-for
  "Given an s/or spec, and a value v, it returns the pred for v"
  [or-spec v]
  ;; TODO check this at the top level
  (s/assert or-spec v)
  (let [form       (s/form or-spec)
        matching-k (first (s/conform or-spec v))
        k->pred    (kwargs->m (rest form))]
    (k->pred matching-k)))
(defmethod pre-serialize* 'clojure.spec.alpha/or
  [spec v]
  (pre-serialize (or-pred-for spec v) v))

(defmethod pre-serialize* 'clojure.spec.alpha/and [spec v]
  (let [preds (seq (rest (s/form spec)))]
    ;; if (empty? preds), return v
    ;; if all preds fail, return v
    ;; otherwise, apply (pre-serialize pred v) for all preds
    (reduce (fn [v p]
              (try
                (pre-serialize p v)
                (catch Exception ex
                  (log/trace ex
                             "catching ex in pre-serialize*, returning v"
                             (merge (ex-data ex) {:v v, :p p}))
                  v)))
            v
            preds)))

(defmethod pre-serialize* :default [_ v] v)

(defn pre-serialize
  "Prepares a value for encoding to Avro.

  For example, it converts `BigDecimal`s to bytes for `:decimal`s,
  `:timestamp-millis` to integers and so on, according to the Avro
  specification."
  [spec-k v]
  (pre-serialize* spec-k v))

;;,------------------------
;;| Avro post-deserialization
;;`------------------------
(declare post-deserialize)

(s/fdef post-deserialize*
        :args (s/cat :spec ::spec
                     :deserialized-value any?))
(defmulti post-deserialize*
  "See post-deserialize.

  NOTE: you shouldn't  need to call this directly, but you can extend this multi to implement
  value post-deserialization for specs that aren't included."
  (fn [spec _v] (spec-variant spec)))

(defmethod post-deserialize* 'scribe.decimal/->decimal-conformer
  [spec v]
  (let [{:keys [precision scale]} (kwargs->m (rest spec))]
    ;; TODO implement fixed
    (decimal/bytes->decimal v precision scale)))

(defmethod post-deserialize* :scribe.avro/enum
  [_ v]
  (keyword v))

(defmethod post-deserialize* 'clojure.spec.alpha/keys
  [spec v]
  (let [form        (s/form spec)
        opts        (kwargs->m (rest form))
        req-un-keys (:req-un opts)]
    ;; TODO opt-un
    (reduce (fn [m k-spec]
              (update m
                      (disqualify-keyword k-spec)
                      (partial post-deserialize k-spec)))
            v
            req-un-keys)))

(defmethod post-deserialize* 'clojure.spec.alpha/coll-of
  [spec v]
  (let [pred (-> spec s/form second)]
    (map (partial post-deserialize pred) v)))

(defn- or-avro-value-spec
  "Given the preds from an s/or form, it returns the spec of the value"
  [or-preds v]
  (let [v-type (-> v type keyword)]
    (->> or-preds
         (filter (partial = v-type))
         first)))
(defmethod post-deserialize* 'clojure.spec.alpha/or
  [spec v]
  (if-let [sn (or-avro-value-spec (or-preds spec) v)]
    (post-deserialize sn v)
    v))

(defmethod post-deserialize* 'clojure.spec.alpha/and [spec v]
  (let [preds (seq (rest (s/form spec)))]
    ;; if (empty? preds), return v
    ;; if all preds fail, return v
    ;; otherwise, apply (post-deserialize pred v) for all preds
    (reduce (fn [v p]
              (try
                (post-deserialize p v)
                (catch Exception ex
                  (log/trace ex
                             "catching ex in post-deserialize*, returning v"
                             (merge (ex-data ex) {:v v, :p p}))
                  v)))
            v
            preds)
    ))

(defmethod post-deserialize* :default [_ v] v)

(defn post-deserialize
  "Processes a deserailized value from Avro.

  For example, it converts bytes to a BigDecimal for `:decimal`s, integers to
  Instants for `:timestamp-millis` and so on, according to the Avro
  specification."
  [spec-k v]
  (post-deserialize* spec-k v))

;;,------
;;| SerDe
;;`------

(defn binary-encoded [spec-k & records]
  (let [s (->avro-schema spec-k)
        rs (map (partial pre-serialize spec-k) records)]
    (apply aa/binary-encoded s rs)))

(defn decode [spec-k source]
  (let [s (->avro-schema spec-k)]
    (post-deserialize spec-k (aa/decode s source))))

;; TODO all all the abracads
