# scribe [![pipeline status](https://gitlab.com/vise890/scribe/badges/master/pipeline.svg)](https://gitlab.com/vise890/scribe/commits/master)

Schema transformation engine for clojure.spec, Avro, ...

## [API Docs](https://vise890.gitlab.io/scribe/)

## Usage

[![Clojars
Project](https://img.shields.io/clojars/v/vise890/scribe.svg)](https://clojars.org/vise890/scribe)

```clojure
;;; project.clj
[vise890/scribe "0.1.0-SNAPSHOT"]
```

```clojure
(ns scribe.readme-test
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as string]
            [clojure.test :refer :all]
            [scribe.avro.core :as savro]
            [scribe.decimal :as decimal]))

(s/def :my.ns/string string?)
(s/def :my.ns/nonBlankString (s/and string? (complement string/blank?)))
(s/def :my.ns/decimal
  (s/and decimal?
         (decimal/->decimal-conformer :scale 3
                                      :precision 5)))
(s/def :my.ns/enum #{:beep :boop})
(s/def :my.ns/collOf (s/coll-of string?))
(s/def :my.ns/recordOne
  (s/keys :req-un [:my.ns/string
                   :my.ns/decimal
                   :my.ns/enum
                   :my.ns/collOf]))
(s/def :my.ns/recordTwo
  (s/keys :req-un [:my.ns/string
                   :my.ns/nonBlankString
                   :my.ns/decimal
                   :my.ns/enum
                   :my.ns/recordOne]))
(s/def :my.ns/or
  (s/or :r1 :my.ns/recordOne
        :r2 :my.ns/recordTwo))

(def record1
  {:string "x7CtK",
   :decimal 0.43M,
   :enum :beep,
   :collOf ["G3" "R7" "xr98"]})

(def record2
  {:string "3G6A",
   :nonBlankString "l20na04",
   :decimal -0.125M,
   :enum :boop,
   :recordOne record1})

(deftest showcase-test

  (s/check-asserts true)

  (testing "generating an avro schema"
    (is (savro/->avro-schema :my.ns/or)))

  (testing "encoding to avro (binary)"
    ;; note that it takes care of serializing decimals to bytes
    (is (bytes? (savro/binary-encoded :my.ns/or record1)))
    (is (bytes? (savro/binary-encoded :my.ns/or record2))))

  (testing "decoding from (binary) avro"
    ;; note that it takes care of deserializing decimals from bytes
    (is (savro/decode :my.ns/or (savro/binary-encoded :my.ns/or record1)))
    (is (savro/decode :my.ns/or (savro/binary-encoded :my.ns/or record2))))

  (testing "round-tripping through avro yields original value"
    (let [value-back (->> record1
                          (savro/binary-encoded :my.ns/or)
                          (savro/decode :my.ns/or))]
      (is (= record1 value-back))))

  (testing "inputs conform to the spec"
    (s/assert :my.ns/recordOne record1)
    (s/assert :my.ns/recordTwo record2)
    (s/assert :my.ns/or record1)
    (s/assert :my.ns/or record2)))
```

## License

Copyright © 2017 Martino Visintin

Distributed under the Eclipse Public License either version 1.0 or (at your
option) any later version.
