(ns zookareg.kafka
  "Utilities for configuring, starting and halting an embedded Kafka broker."
  (:require [integrant.core :as ig]
            [me.raynes.fs :as fs]
            [zookareg.utils :as ut])
  (:import java.io.File
           [kafka.server KafkaConfig KafkaServerStartable]))

(defn ->config
  "Creates a Kafka config from ports and optionally overrides."
  ([ports] (->config ports nil))
  ([{:keys [zookeeper kafka] :as _ports} overrides]
   (merge {:broker.id                        0
           :listeners                        (str "PLAINTEXT://localhost:" kafka)
           :bootstrap.servers                (str "localhost:" kafka)
           :zookeeper.connect                (str "127.0.0.1:" zookeeper)
           :zookeeper-port                   (str zookeeper)
           :log.flush.interval.messages      1
           :auto.create.topics.enable        true
           :group.id                         "consumer"
           :retry.backoff.ms                 500
           :message.send.max.retries         5
           :auto.commit.enable               false
           :log.dir                          (.getAbsolutePath ^File (fs/temp-dir "kafka-log"))
           :offsets.topic.replication.factor 1}
          overrides)))

(defn- ->broker
  "Starts a Kafka broker from a config produced with `->config`."
  [config]
  (let [config (KafkaConfig. (ut/m->properties config))
        server (KafkaServerStartable. config)]
    (.startup server)
    server))

(defn shutdown
  "Stops a Kafka broker."
  [^KafkaServerStartable b]
  (when b
    (.shutdown b)
    (.awaitShutdown b)))

(defmethod ig/init-key ::kafka [_ {:keys [config]}]
  (->broker config))

(defmethod ig/halt-key! ::kafka [_ b]
  (shutdown b))
