(ns dos-installer.executable
  (:require [cheshire.core :as json]
            [clojure.java.io :as io]
            [dos-installer.utils.deps :as utils.deps]
            [dos-installer.utils.files :as utils.files]))

;; constants

(def packr-file-name "packr.json")

(def packr-constants-cfg
  {:platform "windows64"
   :jdk "/app/windows-dependencies/openjdk-11.0.2_windows.zip"
   :output "/app/windows-artifacts"})

(defn packr-cfg []
  packr-constants-cfg)

;; app-cfg data models

(defn app-name
  [app-cfg]
  (:name app-cfg))


;; exec-cfg data models

(defn main-class
  [exec-cfg]
  (:main-class exec-cfg))

(defn resources
  [exec-cfg]
  (:resources exec-cfg))

(defn jvm-args
  [exec-cfg]
  (:jvm-args exec-cfg))

(defn help
  [exec-cfg]
  (:help exec-cfg))

(defn removelibs
  [exec-cfg]
  (:removelibs exec-cfg))

(defn verbose
  [exec-cfg]
  (:verbose exec-cfg))

(defn cachejre
  [exec-cfg]
  (:cachejre exec-cfg))

(defn minimizejre
  [exec-cfg]
  (:minimizejre exec-cfg))

(defn zgc
  [exec-cfg]
  (:use-zgc-if-supported-os exec-cfg))

;; packr-cfg data models

(defn add-main-class
  [packr-cfg exec-cfg]
  (assoc packr-cfg :mainclass (main-class exec-cfg)))

(defn add-resources
  [packr-cfg exec-cfg]
  (assoc packr-cfg :resources (map #(str "/app/" %) (resources exec-cfg))))

(defn add-jvm-args
  [packr-cfg exec-cfg]
  (let [args (jvm-args exec-cfg)]
    (if (empty? args)
      packr-cfg
      (assoc packr-cfg :jvm-args args))))

(defn add-executable-name
  [packr-cfg app-cfg]
  (assoc packr-cfg :executable (app-name app-cfg)))

(defn add-classpath-name
  [packr-cfg jarname]
  (assoc packr-cfg :classpath [(format "/app/%s" jarname)]))

(defn add-removelibs
  [packr-cfg exec-cfg]
  (assoc packr-cfg :removelibs (map #(str "/app/" %) (removelibs exec-cfg))))

(defn add-help
  [packr-cfg exec-cfg]
  (assoc packr-cfg :help (help exec-cfg)))

(defn add-verbose
  [packr-cfg exec-cfg]
  (assoc packr-cfg :verbose (verbose exec-cfg)))

(defn add-cache-jre
  [packr-cfg exec-cfg]
  (assoc packr-cfg :cachejre (cachejre exec-cfg)))

(defn add-minimize-jre
  [packr-cfg exec-cfg]
  (let [minimize (if-let [m (minimizejre exec-cfg)]
                   m
                   "soft")]
    (assoc packr-cfg :minimizejre minimize)))

(defn add-use-zgc-if-supported-os
  [packr-cfg exec-cfg]
  (assoc packr-cfg :useZgcIfSupportedOs (zgc exec-cfg)))

(defn remove-nil
  [packr-cfg]
  (letfn [(check-val [v]
            (if (symbol? v) false
                (or (nil? v) (empty? v))))]
    (into {} (remove (comp check-val second) packr-cfg))))

;; deps data model
(defn packr-content
  [deps]
  (let [{:keys [application executable]} deps
        jarname (utils.deps/jarname deps)]
    (-> (packr-cfg)
        (add-executable-name application)
        (add-classpath-name jarname)
        (add-removelibs executable)
        (add-main-class executable)
        (add-jvm-args executable)
        (add-use-zgc-if-supported-os executable)
        (add-resources executable)
        (add-minimize-jre executable)
        (add-cache-jre executable)
        (add-verbose executable)
        (add-help executable)
        (remove-nil))))

(defn ->json
  [packr-file]
  (json/generate-string packr-file))

(defn packr-json-content
  [deps]
  (-> deps packr-content ->json))

(defn build-packr-json
  [deps user-dir]
  (->> deps
       (utils.deps/executable)
       (packr-json-content)
       (utils.files/paste! user-dir packr-file-name)))
