(ns jax.patcher.compiler.max
  (:refer-clojure :exclude [compile])
  (:require [jax.patcher :as patcher]))

(defn id->str
  [match]
  (if (string? match)
    match
    (if-let [ns (namespace match)]
      (str ns "/" (name match))
      (name match))))

(defn build-router
  [{:keys [router-id]} nodes]
  (let [routes     (->> nodes
                        (reduce
                         (fn [routes {:keys [inlets outlets]}]
                           (into routes
                                 (filter #(= ::patcher/route (:type %)))
                                 (mapcat identity (into inlets outlets))))
                         #{})
                        (vec))
        routetable (->> routes
                        (map-indexed (fn [idx route] [(:id route) idx]))
                        (into {}))
        obj        {:object "route"
                    :type   ::patcher/object
                    :id     router-id
                    :args   (map (comp id->str :id) routes)}]
    [routetable obj]))

(defn build-connection
  [{:keys [router-id]} routetable type id idx connections]
  (map (fn [connection]
         (when connection
           (let [[other-id other-idx] (case (:type connection)
                                        ::patcher/route [router-id (get routetable (:id connection))]
                                        ::patcher/ref [(:id connection) (:idx connection)]
                                        ::patcher/event-channel [(:id connection) 0])]
             (case type
               :input ["script" "connect" other-id other-idx id idx]
               :output ["script" "connect" id idx other-id other-idx]))))
       connections))

(defn build-node-connections
  [env routetable {:keys [inlets outlets id]}]
  (->> (into (map-indexed (partial build-connection env routetable :input id) inlets)
             (map-indexed (partial build-connection env routetable :output id) outlets))
       (mapcat identity)
       (filter identity)))

(defn build-connections
  [env routetable nodes]
  (mapcat (partial build-node-connections env routetable) nodes))

(defn serialize-message-arg
  [arg]
  (if (patcher/route? arg)
    (:init arg)
    arg))

(defn serialize-message-args
  [args]
  (mapv serialize-message-arg args))

(defn build-objects
  [nodes]
  (mapcat (fn [node]
            ;; TODO: something better...
            (let [obj-name (:object node)]
              (case obj-name
                ;; ugh... this is dirty.
                "message"
                (let [args (serialize-message-args (:args node))]
                  [(into ["script" "send" (:id node) "set"] args)
                   ["script" "nth" (:id node) obj-name 1]
                   (into ["script" "newobject" obj-name])])

                ;; majority of objects can be constructed (as we expect) via these two commands to thispatcher
                [["script" "nth" (:id node) obj-name 1]
                 (into ["script" "newobject" obj-name] (serialize-message-args (:args node)))])))
          nodes))

(defmulti compile
          "Compiles a jax object into a collection of instructions to be sent to a thispatcher obj"
          (fn [_env obj] (:type obj)))

(defn build-event-channels
  [{:keys [handler-id]} nodes]
  (let [event-channels (->> nodes
                            (mapcat :outlets)
                            (mapcat identity)
                            (filter #(= ::patcher/event-channel (:type %))))]
    (mapcat
     (fn [event-channel]
       (let [event-type (-> event-channel :event-type name)
             args       (serialize-message-args [event-type "$1" "$2"])]
         [["script" "newobject" "message"]
          ["script" "nth" (:id event-channel) "message" 1]
          (into ["script" "send" (:id event-channel) "set"] args)
          ["script" "connect" (:id event-channel) 0 handler-id 0]]))
     event-channels)))

(defmethod compile ::patcher/instrument
  [{:keys [script-id router-id plugout-id] :as env} {:keys [nodes plugout]}]
  (let [nodes (patcher/->nodes nodes)
        [routetable route-obj] (build-router env nodes)
        nodes (conj nodes route-obj)]
    (into (conj (build-connections env routetable nodes)
                ["script" "connect" (first plugout) (second plugout) plugout-id 0]
                ["script" "connect" script-id 0 router-id 0])
          (into (build-objects nodes)
                (build-event-channels env nodes)))))

(defmethod compile ::patcher/midi-effect
  [{:keys [script-id router-id midiout-id] :as env} {:keys [nodes midiout]}]
  (let [nodes (patcher/->nodes nodes)
        [routetable route-obj] (build-router env nodes)
        nodes (conj nodes route-obj)]
    (into (conj (build-connections env routetable nodes)
                ["script" "connect" (first midiout) (second midiout) midiout-id 0]
                ["script" "connect" script-id 0 router-id 0])
          (into (build-objects nodes)
                (build-event-channels env nodes)))))

(defmethod compile ::patcher/object-set
  [{:keys [script-id router-id] :as env} {:keys [nodes]}]
  (let [nodes (patcher/->nodes nodes)
        [routetable route-obj] (build-router env nodes)
        nodes (conj nodes route-obj)]
    (into (conj (build-connections env routetable nodes)
                ["script" "connect" script-id 0 router-id 0])
          (into (build-objects nodes)
                (build-event-channels env nodes)))))

(defn default-env
  [inst]
  (let [inst-id (some-> inst :id name)]
    {:script-id  "mother"
     :midiin-id  "mother-midiin"
     :midiout-id "mother-midiout"
     :plugin-id  "mother-plugin"
     :plugout-id "mother-plugout"
     :patch-dict "mother-dict"
     :handler-id "mother-handler"
     :router-id  (patcher/genid)
     :inst-id    inst-id}))