(ns jax.impl.inlet
  (:require [clojure.core.async :as async]
            [jax.impl.system :as system]
            [jax.impl.patch :as patch]
            [jax.impl.serdes :as serdes]
            [jax.impl.meta :as meta]
            [jax.impl.runtime :as jax]
            [clojure.tools.logging :as log])
  (:import (java.util UUID)))

(defn process!
  [type value]
  (try
    (case type
      "capture"
      (apply system/handled-captured (serdes/deserialize value))

      "router"
      (async/put! (system/sente-ch) {:event [:jax.inlet/route (serdes/deserialize value)]})

      "clear"
      (jax/clear)

      "restart"
      (patch/restart!)

      "version"
      (log/info meta/version)

      "nrepl"
      (log/info (system/nrepl-port))

      "jetty"
      (log/info (system/jetty-port))

      "console"
      (let [log [(str (UUID/randomUUID)) (serdes/deserialize value)]]
        (async/put! (system/sente-ch) {:event [:jax.inlet/console log]}))

      "web"
      (let [endpoint (format "http://localhost:%s/index.html" (system/jetty-port))]
        (jax/route! "jax/ui" endpoint))

      ;; else, it is a message destined for the end-user
      (let [msg {:type  (keyword type)
                 :value (serdes/deserialize value)}]
        (async/put! (patch/chan) msg)))

    (catch Throwable e
      (log/error "Processing inlet message failed for type" type)
      (.printStackTrace e))))