(ns jax.impl.patch
  (:require [clojure.edn :as edn]
            [jax.impl.classpath :as cp]
            [clojure.java.io :as io]
            [clojure.core.async :as async]
            [clojure.string :as str]
            [clojure.tools.deps.alpha :as deps]
            [clojure.tools.logging :as log]
            [jax.impl.system :as system])
  (:import (java.io File)
           (java.security MessageDigest)))

(defonce ^:private patch
  (atom {}))

(defn running? []
  (some? (seq @patch)))

(def ^String no-patch-loaded
  "No patch has been initialized. Have you loaded a patch?")

(defn instance-id
  []
  (if-let [id (-> patch deref :id)]
    id
    (throw (RuntimeException. no-patch-loaded))))

(defn classpath
  []
  (:classpath @patch))

(defn tmp-dir []
  (if-let [dir (-> patch deref :tmp-dir)]
    dir
    (throw (RuntimeException. no-patch-loaded))))

(defn dir []
  (if-let [dir (-> patch deref :dir)]
    dir
    (throw (RuntimeException. no-patch-loaded))))

(defn state-file [id]
  (io/file (tmp-dir) (instance-id) (str id ".edn")))

(defn slurp-edn
  [s]
  (let [file (io/file s)]
    (if (.exists file)
      (-> file slurp edn/read-string)
      (throw (ex-info (str "File not found: " s) {:file file})))))

(defn chan
  []
  (if-let [ch (-> patch deref :ch)]
    ch
    (throw (RuntimeException. no-patch-loaded))))

(defn sha256
  [string]
  (let [digest (.digest (MessageDigest/getInstance "SHA-256") (.getBytes string "UTF-8"))]
    (apply str (map (partial format "%02x") digest))))

(defn filter-jax-deps
  [deps]
  (let [blacklisted-deps #{'wavejumper/jax
                           'com.cycling74/max
                           'wavejumper/jax-patcher
                           'org.clojure/clojure}]
    (update deps :deps #(into {} (remove (fn [[dep _]] (blacklisted-deps dep))) %))))

(defn deps
  [dir]
  (let [deps  (slurp-edn (io/file dir "deps.edn"))
        paths (map (fn [path] (str (io/file dir path))) (get deps :paths ["src"]))]
    (-> deps
        (filter-jax-deps)
        (deps/resolve-deps nil)
        (deps/make-classpath paths nil))))

(defn load-dep!
  [dep]
  (try
    (log/debug "Adding to classpath" dep)
    (cp/add-classpath dep)
    dep
    (catch Throwable e
      (log/error "Failed to add dep to classpath" dep)
      (.printStackTrace e))))

(defn load-deps!
  [dir]
  (let [deps (remove str/blank? (str/split (deps dir) #":"))]
    (into [] (keep load-dep!) deps)))

(def o (Object.))

(declare restart!)

(defn watch-config!
  [dir]
  (let [jax-edn-file  (io/file dir "jax.edn")
        deps-edn-file (io/file dir "deps.edn")
        close-ch      (async/chan)
        config-loop   (async/go-loop [deps (slurp-edn deps-edn-file)
                                      jax (slurp-edn jax-edn-file)]
                        (async/alt!
                          (async/timeout 500)
                          (let [next-deps (try (slurp-edn deps-edn-file)
                                               (catch Throwable _
                                                 (log/warn "deps.edn malformed, not reloading")
                                                 deps))
                                next-jax  (try (slurp-edn jax-edn-file)
                                               (catch Throwable _
                                                 (log/warn "jax.edn malformed, not reloading")
                                                 jax))]
                            (if (or (not= next-deps deps) (not= next-jax jax))
                              (do (when (not= next-deps deps)
                                    (log/info "deps.edn updated, reloading jax"))
                                  (when (not= next-jax jax)
                                    (log/info "jax.edn updated, reloading jax"))
                                  (try (restart!)
                                       (catch Throwable e
                                         (log/error "Failed to reatart jax")
                                         (.printStackTrace e))))
                              (recur next-deps next-jax)))

                          close-ch
                          nil))]

    {:config-loop config-loop
     :close-ch    close-ch}))

(defn stop!
  []
  (try
    (locking o
      (let [curr-patch @patch]
        (when-not (empty? curr-patch)
          (log/infof "Stopping patch %s" (str (:dir curr-patch)))
          (when-let [close-ch (-> curr-patch :watcher :close-ch)]
            (async/close! close-ch))
          (when-let [ch (:ch curr-patch)]
            (async/close! ch))
          (when-let [stop-fn (:stop-fn curr-patch)]
            (try
              (log/debug "Invoking" (-> curr-patch :config :stop-fn))
              (stop-fn)
              nil
              (catch Throwable e
                (log/error "Exception invoking" (-> curr-patch :config :stop-fn))
                (.printStackTrace e))
              (finally
                (system/broadcast! [:jax/close-patch (-> curr-patch :dir str)])))))))
    (finally
      (reset! patch {}))))

(def config-defaults
  {:buffer-size 1000
   :log-level   :DEBUG
   :splash?     true})

(defn read-jax-config
  [dir]
  (merge config-defaults (slurp-edn (io/file dir "jax.edn"))))

;; Adapted from: https://github.com/Raynes/fs/blob/master/src/me/raynes/fs.clj
(defn home
  ([]
   (io/file (System/getProperty "user.home")))
  ([user]
   (let [homedir  (io/file (System/getProperty "user.home"))
         usersdir (.getParent homedir)]
     (if (empty? user)
       homedir
       (io/file usersdir user)))))

(defn expand-home
  [^String path]
  (if (.startsWith path "~")
    (let [sep (.indexOf path File/separator)]
      (if (neg? sep)
        (home (subs path 1))
        (io/file (home (subs path 1 sep)) (subs path (inc sep)))))
    (io/file path)))

(defn load-patch
  [^String dir]
  (try
    (let [curr-patch @patch
          dir        (expand-home dir)
          tmp-dir    (io/file dir ".jax")]
      (if-not (empty? curr-patch)
        (throw (RuntimeException. (str "Patch already open: " (:dir curr-patch))))
        (locking o
          (log/info "Initializing patch from" (str dir))

          (let [jax-config (read-jax-config dir)
                {:keys [start-fn stop-fn watch-config? buffer-size]} jax-config
                ch         (async/chan (async/sliding-buffer buffer-size))
                classpath  (load-deps! (str dir))]
            (log/debug "Invoking " start-fn)
            (let [start-fn   (requiring-resolve start-fn)
                  stop-fn    (when stop-fn
                               (requiring-resolve stop-fn))
                  watcher    (when watch-config?
                               (watch-config! dir))
                  next-patch {:config    jax-config
                              :start-fn  start-fn
                              :stop-fn   stop-fn
                              :dir       dir
                              :ch        ch
                              :classpath classpath
                              :id        (sha256 (str dir))
                              :tmp-dir   tmp-dir
                              :watcher   watcher}]
              (swap! patch merge next-patch)
              (start-fn ch)
              (system/broadcast! [:jax/load-patch (str dir)])
              next-patch)))))
    (catch Throwable e
      (stop!)
      (throw e))))

(defn restart!
  "Restarts the currently open patch"
  []
  (let [curr-patch @patch]
    (when-not (empty? curr-patch)
      (stop!)
      (load-patch (str (:dir curr-patch))))))

(defn new-patch
  [dir]
  (let [dir (expand-home dir)]
    (if (.exists dir)
      (throw (RuntimeException. (format "Directory %s already exists." (str dir))))
      (with-open [deps  (io/input-stream (io/resource "patch/deps.edn"))
                  jax   (io/input-stream (io/resource "patch/jax.edn"))
                  patch (io/input-stream (io/resource "patch/patch.clj"))]
        (let [patch-file (io/file (io/file dir "src") "patch.clj")]
          (io/make-parents patch-file)
          (io/copy deps (io/file dir "deps.edn"))
          (io/copy jax (io/file dir "jax.edn"))
          (io/copy patch patch-file)
          (log/infof "Created new patch %s" (str dir)))))))