(ns jax.test
  (:require [jax.impl.system :as system]
            [jax.impl.patch :as patch]
            [jax.impl.serdes :as serdes]
            [jax.impl.inlet :as inlet]
            [jax.patcher.test]
            [clojure.java.io :as io]
            [potemkin :as potemkin])
  (:import (jax JaxRuntime$JaxObject)
           (java.io Closeable)))

(potemkin/import-vars
 [jax.patcher.test
  wrap-genid])

(defn jax-object
  "Returns a mocked impl of JaxRuntime$JaxObject"
  [state]
  (proxy [JaxRuntime$JaxObject] []
    (getNREPLPort []
      -1)

    (getSente []
     ;; TODO: add sente here...
      nil)

    (getJettyPort []
      -1)

    (doBang []
      (swap! state update :events #(conj (vec %) [:doBang]))
      nil)

    (doInlet
      ([v]
       (cond
         (string? v)
         (inlet/process! "string" v)

         (double? v)
         (inlet/process! "double" v)

         (instance? Long v)
         (inlet/process! "long" v)

         (integer? v)
         (inlet/process! "int" v)

         (float? v)
         (inlet/process! "float" v)

         :else (throw (RuntimeException. "Unknown type"))))

      ([type args]
       (inlet/process! type args)))

    (doOutlet
      ([v]
       (swap! state update :events #(conj (vec %) [:doOutlet (serdes/deserialize v)]))
       nil)

      ([type xs]
       (swap! state update :events #(conj (vec %) [:doOutlet type (serdes/deserialize xs)]))
       nil))

    (doZap []
      (swap! state update :events #(conj (vec %) [:doZap]))
      nil)))

(defprotocol IMockedRuntime
  (events [_])
  (start! [_]))

(defrecord MockedRuntime [dir api state]
  IMockedRuntime
  (start! [this]
    (when (patch/running?)
      (throw (RuntimeException. "Cannot run a unit test when jax is already running")))

    (assert dir ":dir missing from opts")

    (let [state (atom {})
          api   (jax-object state)]
      (system/register-jax api)
      (patch/load-patch dir)
      (assoc this :api api :state state)))

  (events [_]
    (some-> state deref :events))

  Closeable
  (close [_]
   ;; Don't call runtime/stop! if `start` method hasn't been successfully invoked
    (when (and api events)
      (io/delete-file (patch/tmp-dir) :silently true)
      (patch/stop!))))

(defn mocked-runtime [opts]
  (map->MockedRuntime opts))

(def ^:no-doc o (Object.))

(defmacro with-mocked-runtime
  [[binding opts] & body]
  `(locking o
     (with-open [~binding (start! (mocked-runtime ~opts))]
       (locking patch/o ~@body))))