(ns jax.impl.server
  (:require [ring.adapter.jetty9 :as jetty]
            [jax.impl.system :as system]
            [jax.impl.patch :as patch]
            [jax.impl.meta :as meta]
            [jax.impl.runtime :as runtime]
            [clojure.tools.logging :as log]
            [clojure.core.async :as async]
            [ring.adapter.jetty9.websocket :as ws]
            [clojure.string :as str]
            [ring.middleware.resource :as resource]
            [ring.middleware.file :as file]
            [taoensso.sente.server-adapters.jetty9 :as adapters.jetty9]
            [taoensso.sente :as sente]
            [ring.middleware.defaults :refer [wrap-defaults]]
            [clojure.java.io :as io])
  (:import (java.awt Desktop)
           (java.io Reader BufferedReader InputStreamReader InputStream Closeable IOException)
           (java.net URI)
           (java.nio.charset StandardCharsets)
           (com.pty4j PtyProcess)))

(defn -char-handler
  [^BufferedReader rdr ^Runnable f]
  (let [id     (str (gensym "line-handler*"))
        thread (Thread. f id)]
    (.start thread)
    (reify
      Closeable
      (close [_]
        (.close rdr)
        (.join thread)))))

(defn buffered-reader-open?
  [^BufferedReader reader]
  (try (.ready reader)
       true
       (catch IOException _ false)))

(defmacro char-handler
  [[binding ^Reader reader] & body]
  `(let [r# (BufferedReader. ~reader)
         f# (fn []
              (while (buffered-reader-open? r#)
                (when-let [~binding (.read r#)]
                  ~@body)))]
     (-char-handler r# f#)))

(defn jax-jar
  [cp]
  ;; TODO: better filter fn
  (first (filter #(str/ends-with? % "jax-1.0.0-standalone.jar") cp)))

(defn repl-cmd []
  (if-let [cp (-> (meta/system) :classpath seq)]
    ["java" "-cp" (jax-jar cp) "jax.JaxRepl"
     "--custom-help" "(user/splash)"
     "--eval" "(user/prelude)"
     "--attach" (str "localhost:" (system/nrepl-port))]

    ["lein" "repl" ":connect" (str (system/nrepl-port))]))

(defn repl-handler [_]
  (log/info "New WS connection")
  (try (let [cmd    (into-array String (repl-cmd))
             pty    (PtyProcess/exec cmd (System/getenv) (System/getenv "HOME"))
             os     (.getOutputStream pty)
             is     (.getInputStream pty)
             in-ch  (async/chan)
             out-ch (async/chan)
             out    (char-handler [out (InputStreamReader. ^InputStream is StandardCharsets/UTF_8)]
                      (async/>!! out-ch out))
             in     (async/go-loop []
                      (when-let [val (async/<! in-ch)]
                        (.write os (.getBytes val))
                        (recur)))]
         {:on-connect (fn [ws]
                        (async/go-loop []
                          (when-let [out (async/<! out-ch)]
                            (when (>= out 0)
                              (ws/send! ws (str (char out))))
                            (recur))))
          :on-text    (fn [_ msg]
                        (async/>!! in-ch msg))
          :on-close   (fn [_ x y]
                        (log/debug "Closing ws connection..." x y)
                        (.destroyForcibly pty)
                        (.close out)
                        (.close os)
                        (.close is)
                        (async/close! in)
                        (async/close! in-ch)
                        (async/close! out-ch))})
       (catch Throwable t
         (log/error "WS connection failed")
         (log/info (pr-str (repl-cmd)))
         (.printStackTrace t))))

(defn process-log-message!
  [broadcast-fn event-ctx]
  (broadcast-fn [:jax/console event-ctx]))

(defn send-initial-state!
  [{:keys [console router]} send-fn]
  (let [patch-files (try (file-seq (io/file (patch/tmp-dir) (patch/instance-id)))
                         (catch Throwable _ {}))
        evaled      (->> patch-files
                         (keep (fn [f]
                                 (try (patch/slurp-edn f)
                                      (catch Throwable _))))
                         (keep (fn [x]
                                 (when-let [id (:id x)]
                                   [id x])))
                         (into {}))]
    (let [initial-state {:console console
                         :router  router
                         :system  (meta/system)
                         :evaled  evaled
                         :patch   (when (patch/running?)
                                    (str (patch/dir)))}]
      (send-fn [:jax/initial-state initial-state]))))

(defn open-docs []
  (.browse (Desktop/getDesktop) (URI. (:docs (meta/system)))))

(defn open-github []
  (.browse (Desktop/getDesktop) (URI. (:github (meta/system)))))

(defmulti handle-event (fn [_state _ctx event] (first event)))

(defmethod handle-event :default
  [state _ [event-type _]]
  (log/warnf "Unknown sente message %s" event-type)
  state)

(defmethod handle-event :chsk/uidport-open
  [state {:keys [send-fn]} _]
  (send-initial-state! state send-fn)
  state)

(defmethod handle-event :chsk/uidport-close
  [state _ _]
  state)

(defmethod handle-event :chsk/ws-ping
  [state _ _]
  state)

(defmethod handle-event :jax/route
  [state _ [_ event-ctx]]
  (try (apply runtime/route! event-ctx)
       (catch Throwable e
         (log/errorf e "Failed to route message %s" event-ctx)))
  state)

(defmethod handle-event :jax.inlet/console
  [state {:keys [broadcast-fn]} [_ event-ctx]]
  (process-log-message! broadcast-fn event-ctx)
  (update state :console conj event-ctx))

(defmethod handle-event :jax.inlet/route
  [state {:keys [broadcast-fn]} [_ [id val]]]
  (broadcast-fn [:jax/route [(keyword id) val]])
  (assoc-in state [:router (keyword id)] val))

(defmethod handle-event :jax.runtime/clear
  [state _ [_ event-ctx]]
  (runtime/clear event-ctx)
  state)

(defmethod handle-event :jax.runtime/clear-all
  [state _ _]
  (runtime/clear)
  state)

(defmethod handle-event :client/error
  [state _ [_ event-ctx]]
  (log/error "Client-side error" (pr-str event-ctx))
  state)

(defmethod handle-event :docs/open
  [state _ _]
  (open-docs)
  state)

(defmethod handle-event :docs/open-github
  [state _ _]
  (open-github)
  state)

(defn event-receiver
  [broadcast-fn send-fn ch]
  (async/go-loop [state {:console [] :router {}}]
    (when-let [message (async/<! ch)]
      (let [client-id  (:client-id message)
            send-fn    (partial send-fn client-id)
            ctx        {:broadcast-fn broadcast-fn
                        :send-fn      send-fn}
            next-state (try (handle-event state ctx (:event message))
                            (catch Throwable e
                              (log/errorf e "Error processing messsage %s" (-> message :event first))
                              state))]
        (recur next-state)))))

(defn broadcast!
  [{:keys [send-fn connected-uids]} message]
  (doseq [uid (:ws @connected-uids)]
    (send-fn uid message)))

(def chsk-defaults
  {:params    {:urlencoded true
               :multipart  true
               :nested     true
               :keywordize true}
   :responses {:not-modified-responses true
               :absolute-redirects     true
               :content-types          false
               :default-charset        "utf-8"}})

(defn sente
  []
  (let [sente        (sente/make-channel-socket! (adapters.jetty9/get-sch-adapter)
                                                 {:csrf-token-fn nil
                                                  :user-id-fn    #(-> % :params :client-id)})
        broadcast-fn (partial broadcast! sente)]
    (assoc sente
           :ch-recv-loop (event-receiver broadcast-fn (:send-fn sente) (:ch-recv sente))
           :broadcast-fn broadcast-fn)))

(def ^:dynamic *dev-mode* true)
(def ^:dynamic *dev-public-dir* "/Users/thomascrowley/Code/clojure/jax/jax/resources/public")

(defn server
  [sente]
  (let [sente-handler (:ajax-get-or-ws-handshake-fn sente)]
    (jetty/run-jetty
     (if *dev-mode*
       (file/wrap-file (constantly nil) *dev-public-dir* {:allow-symlinks? true})
       (resource/wrap-resource (constantly nil) "public"))
     {:port                     0
      :join?                    false
      :websockets               {"/repl" repl-handler
                                 "/chsk" (wrap-defaults sente-handler chsk-defaults)}
      :ws-max-idle-time         (* 60 60 24 1000)
      :ws-max-text-message-size (* 1024 1024 4)
      :allow-null-path-info     true})))