(ns {{name}}.handler
  (:use [ring.util.servlet]
        [ring.middleware.multipart-params]
        [ring.middleware.params :only [wrap-params]]
        [ring.middleware.format-params :only [wrap-restful-params]]
        [ring.middleware.format-response :only [wrap-format-response wrap-restful-response wrap-json-response]]
        [compojure.core]
        [cljs.net.remote.service])
  (:require
   [clojure.tools.logging :as log]
   [ring.util.response :as resp]
   [noir.util.middleware :as noir-mid]
   [compojure.route :as route]
   [compojure.response :as response]
   [cemerick.friend :as friend]
   [net.cgrand.enlive-html :as html]
   [{{name}}.modules.auth-handler :as auth]
   [{{name}}.modules.user-handler :as user]
   [{{name}}.modules.home-handler :as home]
   [{{name}}.app-service.upload-handler :as upload]
   [{{name}}.app-service.report :as report]))


(defn remote-handler [app-context]
  (let [r (routes
           (REMOTE POST "/_fetch" [:as request]
                   (fn [f] (f request {:request request
                                      :context app-context}))))]
    [(context "/remote" [] r)]))


(html/deftemplate index "public/index.html" [{:keys [title]}]
  [:head :title] (html/content title)
  [:script html/any-node] (html/replace-vars {:name "world" }))


(defn app-routes [app-context]
  (concat
   [(GET "/ping" [] (resp/response "Ping success"))]
   (remote-handler app-context)
   [(GET "/" [] (resp/response (apply str (index {:title "Index page"}))))]
   (auth/handler app-context)
   (user/handler app-context)
   (report/handler app-context)
   (upload/handler app-context)
   (home/handler app-context)))


(defn wrap-request-log [app]
  (fn [{:keys [request-method uri] :as req}]
    (let [resp (app req)
          _ (log/info (-> req
                          (dissoc :body)
                          (dissoc :credential-fn)
                          (dissoc :cemerick.friend/auth-config)
                          (dissoc :unauthenticated-handler)
                          (dissoc :headers)))]
      resp)))


(defn wrap-exceptions [app]
  (fn [request]
    (try
      (app request)
      (catch Exception e
        (log/error e "System got Exception")
        {:status 500
         :headers {"Content-Type" "text/html"}
         :body "Exception in server"}))))


(defn handler [app-context]
  (let [v (concat [(route/resources "/")]
                  (app-routes app-context)
                  [(route/not-found "Not Found")])
        r (->
           (apply routes v )
           (wrap-params)
           (auth/auth-warp (auth/auth-validation app-context ))
           (wrap-restful-params)
           (wrap-restful-response)
           (wrap-exceptions)
           (wrap-request-log))]
    (noir-mid/war-handler (noir-mid/app-handler
                           [r]
                           :middleware []))))
