(ns {{name}}.db
  (:use
   [datomic.api :only [q db] :as d]
   [{{name}}.core.lifeCycle])
  (:import com.mchange.v2.c3p0.ComboPooledDataSource)
  (:require
   [clojure.tools.logging :as log]
   [clojure.java.jdbc :as jdbc]
   [clojure.java.jdbc :as sql]
   [clojure.java.io :as io]
   [{{name}}.modules.user :as u]
   [{{name}}.core.db-util :as db-util]
   [{{name}}.modules.user-util :as u-util]))

        ;sudo su - postgres
        ;psql zme
        ;stop logout

(defn ^:private pool [config]
  (log/info "Creating new database connection pool")
  {:datasource
   (doto (ComboPooledDataSource.)
     (.setDriverClass (:classname config))
     (.setJdbcUrl (str "jdbc:" (:subprotocol config) ":" (:subname config)))
     (.setUser (:user config))
     (.setPassword (:password config))
     (.setMaxPoolSize 6)
     (.setMinPoolSize 1)
     (.setInitialPoolSize 1))})


(defn ^:private init-db [db-conn db]
  (log/info "Init database, please wait...")
  (if-let [dd (db-util/create-data-defination @db)]
    (try
      (apply sql/db-do-commands @db-conn false dd)
      (catch Exception e
        (log/error e))))

  (if-let [dd (db-util/init-db @db)]
      (try
        (apply sql/db-do-commands @db-conn false dd)
        (catch Exception e
          (log/error e))))
  (log/info "Init database is Done"))



(defn ^:private clean-db [db-conn db]
  (log/info "Cleaning db, please wait...")
  (if-let [dd (db-util/drop-data-defination @db)]
      (try
        (apply sql/db-do-commands @db-conn false dd)
        (catch Exception e
          (log/error e)))))


(defrecord SqlDB []
  LifeCycle
  (start [this comp]
    (let [db-file (get-in comp [:config :config :db :file-name])
          db-config (get-in comp [:config :config :db :config])
          db (delay (db-util/load-query db-file))
          db-conn (delay (pool db-config))]
      (log/info "SQL Database connection is opening, please wait...")
      (init-db db-conn db)
      (-> this
          (assoc :db db)
          (assoc :conn db-conn))))
  (stop [this]
    (log/info "SQL Database connection is closeing, please wait...")
    (clean-db (:conn this) (:db this))
    (assoc this :conn nil)))


(defn load-films [context]
  (let [q (db-util/get-query @(get-in context [:sqldb :db]) :films)]
    (jdbc/query @(get-in context [:sqldb :conn]) [q] :as-arrays? true )))



;Datomic

(defn load-schema-tx [file-name]
      (->  file-name
           (io/resource )
           (slurp )
           (read-string )) )

(defn connection [uri schema-tx]
  (when (d/create-database uri)
    @(d/transact (d/connect uri) schema-tx))
  (d/connect uri))


(defn delete-db [uri]
  (d/delete-database uri))


(defrecord Datomic []
  LifeCycle
  (start [this comp]
    (println comp)
    (let [ schema-tx (load-schema-tx (get-in comp [:config :config :datomic :file-name]))
           uri (get-in comp [:config :config :datomic :uri])
           conn  (connection uri schema-tx)]
      (log/info "Datomic is starting, please wait...")
      (u/insert-all conn (u-util/get-random-data 25 ))
      (log/info (str "Total data" (count (u/find-all conn 1 15 ))))
      (-> this
          (assoc :uri uri)
          (assoc :conn conn))))
  (stop [this]
    (log/info "Datomic is stoping, please wait...")
    (delete-db (get this :uri))
    (assoc this :conn nil)))
