(ns {{name}}.core.db-util
    (:require
     [clojure.java.io :as io]))


(defn bind-error [f [val err]]
  (if (nil? err)
    (f val)
    [nil err]))


(defmacro err->> [val & fns]
  (let [fns (for [f fns] `({{name}}.core.db-util/bind-error ~f))]
    `(->> [~val nil]
          ~@fns)))


(defn load-db-file [file-name]
  (let [sql-str (->
                 file-name
                 (io/resource)
                 (slurp)
                 (clojure.string/replace #"--|/\*|\*/|\n" ""))
        sql-str-list (str "(" sql-str ")")
        sql-list (read-string sql-str-list)]
    (reduce (fn [v c]
              (assoc v (keyword (:name c)) c))
            {} sql-list)))


(defn apply-params [m q]
  (let [exp-keys (set (re-seq #"\w*:\w+" q))
        inp-keys (set (map str (keys m)))]
    (if (clojure.set/superset? inp-keys exp-keys)
      [(reduce (fn [acc [k v]]
                 (clojure.string/replace acc (str k) (str v)))
               q m) nil]
      [nil (format "Parameter missing %s" (pr-str exp-keys)) ])))


(defn get-sql [db name]
  (let [ name (if (keyword? name) name
                  (keyword name))
        q  (get-in db [name :sql])]
    (if-not q
      [nil (format "Could not find %s sql statement" name ) ]
      (cond
       (or (= name :create-dd)
           (= name :drop-dd)
           (= name :init-db))
       [(clojure.string/split q #";") nil]
       :else
       [(clojure.string/replace q #";" "") nil]))))


(defn get-sql-with-params [db name & [params]]
  (err->> name
          (partial get-sql db)
          (partial apply-params params)))
