(ns {{name}}.db
    (:use
     [datomic.api :only [q db] :as d]
     [{{name}}.core.lifeCycle])
    (:import com.mchange.v2.c3p0.ComboPooledDataSource)
    (:require
     [clojure.tools.logging :as log]
     [clojure.java.jdbc :as jdbc]
     [clojure.java.jdbc :as sql]
     [clojure.java.io :as io]
     [{{name}}.modules.user.auth-handler :as u]
     [{{name}}.core.db-util :as db-util]
     [{{name}}.modules.user.user-util :as u-util]))

                                        ;sudo su - postgres
                                        ;psql zme
                                        ;stop logout

(defn ^:private pool [config]
  (log/info "Creating new database connection pool")
  {:datasource
   (doto (ComboPooledDataSource.)
     (.setDriverClass (:classname config))
     (.setJdbcUrl (str "jdbc:" (:subprotocol config) ":" (:subname config)))
     (.setUser (:user config))
     (.setPassword (:password config))
     (.setMaxPoolSize 6)
     (.setMinPoolSize 1)
     (.setInitialPoolSize 1))})


(defn ^:private init-db [db-conn db]
  (log/info "Init database, please wait...")
  (if-let [[dd _] (db-util/get-sql @db :create-dd)]
    (try
      (apply sql/db-do-commands @db-conn false dd)
      (catch Exception e
        (log/error e))))

  (if-let [[dd _] (db-util/get-sql @db :init-db)]
    (try
      (apply sql/db-do-commands @db-conn false dd)
      (catch Exception e
        (log/error e))))
  (log/info "Init database is Done"))



(defn ^:private clean-db [db-conn db]
  (log/info "Cleaning db, please wait...")
  (if-let [[dd _] (db-util/get-sql @db :drop-dd)]
    (try
      (apply sql/db-do-commands @db-conn false dd)
      (catch Exception e
        (log/error e)))))


(defrecord SqlDB []
  LifeCycle
  (start [this comp]
    (let [db-file (get-in comp [:config :config :db :file-name])
          db-config (get-in comp [:config :config :db :config])
          db (delay (db-util/load-db-file db-file))
          db-conn (delay (pool db-config))]
      (log/info "SQL Database connection is opening, please wait...")
      (init-db db-conn db)
      (-> this
          (assoc :db db)
          (assoc :conn db-conn))))
  (stop [this]
    (log/info "SQL Database connection is closeing, please wait...")
    (clean-db (:conn this) (:db this))
    (assoc this :conn nil)))


                                        ;Datomic

(defn ^:private load-schema-tx [file-name]
  (->  file-name
       (io/resource )
       (slurp )
       (read-string )) )

(defn ^:private connection [uri schema-tx]
  (when (d/create-database uri)
    @(d/transact (d/connect uri) schema-tx))
  (d/connect uri))


(defn ^:private delete-db [uri]
  (d/delete-database uri))


(defrecord Datomic []
  LifeCycle
  (start [this comp]
    (let [ schema-tx (load-schema-tx (get-in comp [:config :config :datomic :file-name]))
          uri (get-in comp [:config :config :datomic :uri])
          conn  (connection uri schema-tx)]
      (log/info "Datomic is starting, please wait...")
      (u/insert-all conn (u-util/get-random-data 25 ))
      ;(log/info (str "Total data" (count (u/find-all conn "1"  ))))
      (-> this
          (assoc :uri uri)
          (assoc :conn conn))))
  (stop [this]
    (log/info "Datomic is stoping, please wait...")
    (delete-db (get this :uri))
    (assoc this :conn nil)))
