(ns {{name}}.routes
    (:use
     [ring.util.servlet]
     [ring.middleware.multipart-params]
     [ring.middleware.params :only [wrap-params]]
     [ring.middleware.format-params :only [wrap-restful-params]]
     [ring.middleware.format-response :only [wrap-format-response wrap-restful-response wrap-json-response]]
     [compojure.core]
     [cljs.net.remote.service]
     [{{name}}.core.lifeCycle])
    (:require
     [clojure.tools.logging :as log]
     [ring.util.response :as resp]
     [noir.util.middleware :as noir-mid]
     [compojure.route :as route]
     [compojure.response :as response]
     [cemerick.friend :as friend]
     (cemerick.friend [workflows :as workflows]
                       [credentials :as creds])
     [{{name}}.views :as v]
     [{{name}}.modules.query-handler :as query]
     [{{name}}.modules.remote-handler :as remote]
     [{{name}}.modules.user.auth-handler :as auth]
     [{{name}}.modules.user.user-handler :as user]
     [{{name}}.modules.home.home-handler :as home]
     [{{name}}.modules.upload.upload-handler :as upload]
     [{{name}}.modules.reporting.report-handler :as report]))


(defn ^:private wrap-request-log
  "Log http request"
  [app]
  (fn [{:keys [request-method uri] :as req}]
    (let [resp (app req)]
      (log/info (-> req
                    (dissoc :body)
                    (dissoc :credential-fn)
                    (dissoc :cemerick.friend/auth-config)
                    (dissoc :unauthenticated-handler)
                    (dissoc :headers)))
      resp)))


(defn ^:private wrap-exceptions
  "Catch all system exception "
  [app]
  (fn [request]
    (try
      (app request)
      (catch Exception e
        (log/error e "System got Exception")
        {:status 500
         :headers {"Content-Type" "text/html"}
         :body "Exception in server"}))))


(defn ^:private warp-auth
  "For authentication system "
  [app app-context]
  (let [ u {:username "admin"
            :password (creds/hash-bcrypt "admin")}
         credential-fn (fn [v]
                         (if (= (:username u ) v)
                           u))]
    (friend/authenticate app
                         {:credential-fn (partial creds/bcrypt-credential-fn credential-fn)
                          :workflows [(workflows/interactive-form)]
                          :login-uri "/login"
                          :unauthorized-redirect-uri "/login"
                          :default-landing-uri "/" })))



(defn ^:private app-routes
  "Routes for application. "
  [app-context]
  (routes
    (GET "/" [](friend/authenticated (resp/response (v/index {:title "Index page"}))))
    (GET "/ping" [](resp/response "Ping success"))
    (route/resources "/")
    (GET "/login" [:as {params :params} ]
                    (if (friend/current-authentication)
                      (resp/redirect "/")
                      (v/login "Login" params)))
    (friend/logout (ANY "/logout" request (resp/redirect "/login")))
    (context "/query" [] (query/query-routes app-context))
    (context "/user" [] (friend/authenticated (user/user-routes app-context)))
    (context "/home" [] (home/home-routes app-context))
    (context "/remote" [] (remote/remote-routes app-context))
    (context "/report" [] (report/report-routes app-context))
    (context "/upload" [] (upload/upload-routes app-context))
    (context "/auth" [] (auth/auth-routes app-context))
    (route/not-found "Not Found")))


(defn ^:private handler
  "Apply middleware "
  [app-context]
  (let [r (->
           (app-routes app-context)
           (wrap-params)
           (warp-auth app-context)
           (wrap-restful-params)
           (wrap-restful-response)
           (wrap-exceptions)
           (wrap-request-log))]
    (noir-mid/war-handler (noir-mid/app-handler
                           [r]
                           :middleware []))))


(defrecord Web []
  LifeCycle
  (start [this comp]
    (log/info "Web is starting, please wait...")
    (assoc this :handler (handler comp)))
  (stop [this]
    (log/info "Web is stoping, please wait...")
    (assoc this :handler nil)))
