(ns webnf.async-servlet.websocket
  (:require [clojure.core.async :refer [go <! >! <!! >!! close! alt!]]
            [clojure.java.io :as io]
            [clojure.tools.logging :as log])
  (:import (webnf UpgradeHandler)
           (java.io Writer InputStream File)
           (java.nio ByteBuffer)
           (java.nio.channels Channels WritableByteChannel)
           (javax.servlet ServletInputStream ServletOutputStream ReadListener)
           (javax.servlet.http WebConnection)
           java.security.MessageDigest
           java.util.Base64))

(defn add-read-handler! [conn open read-ch ^ServletInputStream in]
  (let [in-channel (Channels/newChannel in)]
    (.setReadListener
     in (reify ReadListener
          (onDataAvailable [_]
            (while (and (:read @open)
                        (.isReady in))
              (let [buf (ByteBuffer/allocate (* 1024 4))]
                (while (and (:read @open)
                            (.isReady in)
                            (pos? (.capacity buf)))
                  (.read in-channel buf))
                (when (:read @open)
                  (>!! read-ch buf)))))
          (onAllDataRead [_]
            (close! read-ch)
            (vswap! open disj :read))
          (onError [this t]
            (log/error t "Error in websocket ReadListener")
            (.onAllDataRead this)
            (.close conn))))))

(defn value-writer [^ServletOutputStream out ^Writer writer ^WritableByteChannel chan open]
  (fn write! [val]
    (when (:write @open)
      (cond
        (string? val) (.print writer ^String val)
        (seq? val) (doseq [chunk val]
                     (.print writer (str chunk)))
        (instance? InputStream val) (io/copy val out)
        (instance? File val) (io/copy (io/input-stream val) out)
        (instance? ByteBuffer val) (.write chan val)
        nil nil
        :else (throw (ex-info "Unrecognized value" {:value val}))))))

(defn add-write-handler! [conn open write-ch ^ServletOutputStream out]
  (let [writer (io/writer out)
        channel (Channels/newChannel out)
        write! (value-writer out writer channel open)]
    (go
      (try
        (loop []
          (when (:write @open)
            (when-let [first-out-val (<! write-ch)]
              (when (:write @open)
                (write! first-out-val)
                (loop []
                  (when (:write @open)
                    (when-let [out-val (alt! write-ch ([out-val] out-val)
                                             :default ::none-here)]
                      (when (:write @open)
                        (if (= ::none-here out-val)
                          (.flush writer)
                          (do (write! out-val)
                              (recur)))))))
                (recur)))))
        (catch Exception e
          (log/error e "Error in websocket writer")
          (.close conn))
        (finally
          (vswap! open disj :write)
          (.close writer))))))

(defn initialize! [^UpgradeHandler handler ^WebConnection conn]
  (log/debug "Initializing websocket connection" handler conn)
  (add-read-handler! conn (.-openVolatile handler) (.-readChannel handler) (.getInputStream conn))
  (add-write-handler! conn (.-openVolatile handler) (.-writeChannel handler) (.getOutputStream conn)))

(defn destroy [^UpgradeHandler handler]
  (log/debug "Destroying websocket connection" handler)
  (vreset! (.-openVolatile handler) #{})
  (close! (.-readChannel handler)))

(def magic-cookie "258EAFA5-E914-47DA-95CA-C5AB0DC85B11")

(defn- sha1-base64 [s]
  (let [md (MessageDigest/getInstance "SHA-1")]
    (.. Base64 getEncoder (encodeToString (.digest md (.getBytes ^String s))))))

(defn accept-key [key]
  (sha1-base64 (str key magic-cookie)))
