(ns webnf.base.partial-application
  "A pendant for partial, for applying partial arguments _after_ the first one.

  Examples:

  ((pretial conj 1 2 3 4)
   [ 0 ])  ;; => [0 1 2 3 4 5]

  ((pretial conj 1 2 3 4)
   #{ 0 }) ;; => #{0 1 4 3 2}

  (update {:a [0]} :a ap
   (pretial conj 1 2 3)
   (pretial subvec 2)) ;; => {:a [2 3]}"
  (#?(:clj :require :cljs :require-macros)
   [webnf.base.unroll :refer [defunrolled]]
   #?(:cljs [webnf.base.partial-application :refer [or*]])))

#?(:clj (defmacro or* "Variant of or, that only skips nils"
          [v d]
          `(let [v# ~v] (if (nil? v#) ~d v#))))

;; ### pretial -- partial for the first param

(defn pretial* [f & args]
  (fn [o] (apply f o args)))

(defunrolled pretial
  "partial application for arguments after the first.
   i.e. something that works like partial for clojure-style update functions"
  :min 1
  :more-arities ([args] (apply pretial* args))
  ([f] f)
  ([f & args] `(fn* [o#] (~f o# ~@args))))

;; ### ap -- start chains of pretials

;; This is the entry point for update functions
;; it's like a reverse comp, that fits in the update-fn slot:
;; (update-in x [y z] ap
;;            (pretial assoc :a :b)
;;            (pretial dissoc :c))

(defn ap* [x & fs]
  (reduce #((or* %2 identity) %1) x fs))

(defunrolled ap
  "Compose and call pretials. This can be used to start a pretial chain from an update slot.

   Example:
    (ap {:good 0 :bad 1}
        (pretial update :good inc)
        (pretial update :bad dec))"
  :min 1
  :more-arities ([args] (apply ap* args))
  ([x & fs] (reduce  #(list `(or* ~%2 identity) %1) x fs)))

;; ### rcomp seems to naturally fall out

(defn rcomp* [& fs]
  (apply pretial* ap* fs))

(defunrolled rcomp
  "A reverse comp, threading its result from left to right"
  :more-arities ([args] (apply rcomp* args))
  ([& fs] `(pretial ap ~@fs)))
