(ns webnf.base.string
  #?(:cljs (:import  (goog.string StringBuffer)))
  #?(:cljs (:require [cljs.pprint :refer [pprint]]))
  #?(:clj  (:require [webnf.base.autoload :refer [autoload]])))

(declare append!)

(defn string-builder
  "use with append!"
  ([]                 (string-builder 32 ""))
  ([size-or-init-str] (if (string? size-or-init-str)
                        (string-builder (+ 16 (count size-or-init-str)) size-or-init-str)
                        (string-builder size-or-init-str         "")))
  ([size init-str]    #?(:clj  (-> (StringBuilder. size)
                                   (append! init-str))
                         :cljs (StringBuffer. init-str))))

(defn append!
  "Reducer function to append to a string-builder instance
   completing arity returns string from builder"
  ([]          (string-builder))
  ([sb]        #?(:clj  (.toString ^StringBuilder sb)
                  :cljs (.toString ^StringBuffer  sb)))
  ([sb s]      #?(:clj  (.append   ^StringBuilder sb (str s))
                  :cljs (.append   ^StringBuffer  sb (str s))))
  ([sb s & ss] (reduce append!
                       (append! sb s)
                       ss)))

#?(:clj (autoload clojure.pprint/pprint))

(defn pprint-str
  "Return value pretty-printed into a string.
   Allows for clojure.pprint/*print-right-margin* to be passed as second argument"
  ([o] (with-out-str (pprint o)))
  ([o right-margin]
   #?(:clj (require 'clojure.pprint))
   (binding [#?(:clj  clojure.pprint/*print-right-margin*
                :cljs cljs.pprint/*print-right-margin*)
             right-margin]
     (pprint-str o))))

(defn into-str
  "Like (partial apply str), but with the possibility of transducing
   the arguments."
  ([s strs] (into-str s identity strs))
  ([s xf strs]
   (transduce xf append!
              (string-builder (+ (count s)
                                 (* (count strs) 8))
                              s)
              strs)))

(def character-quoter
  #?(:clj  (memoize  (fn [q e]
                       (eval
                        (if (= q e)
                          `(fn [^StringBuilder sb# ch#]
                             (case ch#
                               ~e (.. sb# (append (char ~e)) (append (char ~e)))
                               (.append sb# ch#)))
                          `(fn [^StringBuilder sb# ch#]
                             (case ch#
                               ~e (.. sb# (append (char ~e)) (append (char ~e)))
                               ~q (.. sb# (append (char ~e)) (append (char ~q)))
                               (.append sb# ch#)))))))
     :cljs (fn [q e] (fn [sb ch]
                       (cond
                         (= ch e) (.. sb (append e) (append e))
                         (= ch q) (.. sb (append e) (append q))
                         :else (.append sb ch))))))

(defn str-quote
  "Quotes string with configurable quote and escape character (default \" and \\)"
  ([v] (str-quote v \" \\))
  ([v q] (str-quote v q \\))
  ([v q e]
   (let [s  (str v)
         s* (-> (string-builder (+ (count s) 6) "")
                (append! q)
                (as-> sb (reduce (character-quoter q e)
                                 sb s))
                (append! q)
                str)]
     (if (= s s*)
       s s*))))
