(ns webnf.base.util
  (:refer-clojure :exclude [update-in])
  (#?(:clj :require :cljs :require-macros)
   [webnf.base.unroll :refer [defunrolled]]
   [webnf.base.string :refer [pprint-str]])
  (:require [clojure.string :as str]
            #?@(:clj  [[webnf.base.cljc :refer [defmacro*]]
                       [clojure.tools.logging :as log]]
                :cljs [[webnf.base.logging :as log :include-macros true]])))

(def conjv (fnil conj []))
(def conjs (fnil conj #{}))
(def conjm (fnil conj {}))
(def conjq (fnil conj #?(:clj  clojure.lang.PersistentQueue/EMPTY
                         :cljs [] #_FIXME)))

(defn update-in
  "Version of {clojure,cljs}.core/update-in, fixed for empty paths
  see http://dev.clojure.org/jira/browse/CLJ-1623"
  [m ks f & args]
  (if-let [[k & ks*] (seq ks)]
    (assoc m k (apply update-in (get m k) ks* f args))
    (apply f m args)))

(defmacro condas->
  "chains of (cond-> x, p? (as-> x e), q? (as-> x f),,,)"
  [expr as & test-exprs]
  `(as-> ~expr ~as ~@(for [[t e] (partition 2 test-exprs)]
                       `(if ~t ~e ~as))))

#?
(:clj
 (do
   (defmacro* squelch
     "Eval body with a handler for Exception that returns a default expression val.
  Logs exceptions on trace priority."
     [val & body]
     :clj
     `(try ~@body (catch Exception e#
                    (let [val# ~val]
                      (log/trace e# "during execution of"
                                 ~(pprint-str `(try ~@body ~'(catch Exception e ...)))
                                 "\n used replacement value:" val#)
                      val#)))
     :cljs
     `(try ~@body (catch js/Error e#
                    (let [val# ~val]
                      (webnf.base.logging/trace e# "during execution of"
                                                ~(pprint-str `(try ~@body ~'(catch js/Error e ...)))
                                                "\n used replacement value:" val#)
                      val#))))
   (defmacro* forcat
     "Concat the return value of a for expression"
     [bindings body]
     :clj  `(apply concat (for ~bindings ~body))
     :cljs `(cljs.core/apply cljs.core/concat (cljs.core/for ~bindings ~body)))

   (defmacro static-case
     "Variant of case where keys are evaluated at compile-time
   WARNING: only use this for dispatch values with stable hashes,
     like edn literals, java Enums, ..."
     [val & cases]
     `(case ~val
        ~@(forcat [[field thunk] (partition 2 cases)]
                  [(eval field) thunk])
        ~@(when (odd? (count cases))
            [(last cases)])))))

(defn scat
  "Returns a function taking a seq on which f is applied.
   To [scat]ter is an antonym of to [juxt]apose."
  [f]
  #(apply f %1))

(defn to-coll
  "Ensure that seq can be called on a value. If value is not a coll
  and not nil, it is put into an empty collection"
  [v]
  (if (or (nil? v)
          #?(:clj (instance? java.util.Collection v)
             :cljs (coll? v)))
    v (cons v nil)))

(defn update!* [tm k f & args]
  (assoc! tm k (apply f (get tm k) args)))

(defunrolled update!
  :min 3
  :more-arities ([args] (apply update!* args))
  ([tm k f & args]
   `(let [tm# ~tm
          k# ~k]
      (assoc! tm# k# (~f (get tm# k#) ~@args)))))

(comment
  ;; experimental transducer, that would hard-code a for-step in terms
  ;; of reduce. That could be used to peel a lazy-seq layer, but the
  ;; complexity doesn't seem worth it
  (defmacro forcat-xf [value [binding expr] & body]
    `(fn [xf#]
       (fn
         ([s#] (xf# s#))
         ([s# ~value]
          (reduce (fn [s# ~binding]
                    (xf# s# (do ~@body)))
                  s# ~expr))))))

#?(:clj
   (defmacro* deprecated-alias
     "Define an alias for a function in another namespace"
     [alias target]
     :clj  `(defn ~alias [~'& args#]
              (log/warn "Function" ~(resolve alias) "is DEPRECATED. Please use" ~(resolve target) "instead!")
              (apply ~target args#))
     :cljs `(defn ~alias [~'& args#]
              (webnf.base.logging/warn "Function" ~(pr-str (resolve alias)) "is DEPRECATED. Please use" ~(pr-str (resolve target)) "instead!")
              (apply ~target args#))))
