;; Copyright (c) Frank Siebenlist. All rights reserved.
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file COPYING at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns cljs-uuid.core
  "ClojureScript micro-library with an implementation of a type 4, random UUID generator compatible with RFC-4122 and cljs.core/UUID (v4), a getter function to obtain the uuid string representation from a UUID-instance (to-str), a to-str conformance validating predicate (uuid?), and a UUID factory from to-str with conformance validation (to-uuid)."
  (:require [goog.string.StringBuffer]))

;; see https://gist.github.com/4159427 for some background


;; Future UUID-implementations may chose a different internal representation of the UUID-instance
;; The trivial to-str function hides those UUID-internals.
;; Further motivation for to-str are related to interop thru json or with existing databases.

(defn to-str
  "(to-str a-uuid)  =>  uuid-str
  Arguments and Values:
  a-uuid --- a cljs.core/UUID instance.
  uuid-str --- returns a string representation of the UUID instance
  Description:
  Returns the string representation of the UUID instance in the format of,
  \"xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx\" similarly to java.util.UUID/toString.
  Note that this is different from cljs.core/UUID's EDN string-format.
  Examples:
  (def u (v4))  =>  #uuid \"305e764d-b451-47ae-a90d-5db782ac1f2e\"
  (to-str u) => \"305e764d-b451-47ae-a90d-5db782ac1f2e\""
  [a-uuid]
  (str (.-uuid a-uuid)))


(defn v4
  "(v4)  =>  new-uuid
  Arguments and Values:
  new-uuid --- new type 4 (pseudo randomly generated) cljs.core/UUID instance.
  Description:
  Returns pseudo randomly generated UUID,
  like: xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx as per http://www.ietf.org/rfc/rfc4122.txt.
  Examples:
  (v4)  =>  #uuid \"305e764d-b451-47ae-a90d-5db782ac1f2e\"
  (type (v4)) => cljs.core/UUID"
  []
  (letfn [(f [] (.toString (rand-int 16) 16))
          (g [] (.toString  (bit-or 0x8 (bit-and 0x3 (rand-int 15))) 16))]
    (UUID. (.toString (.append (goog.string.StringBuffer.)
       (f) (f) (f) (f) (f) (f) (f) (f) "-" (f) (f) (f) (f)
       "-4" (f) (f) (f) "-" (g) (f) (f) (f) "-"
       (f) (f) (f) (f) (f) (f) (f) (f) (f) (f) (f) (f))))))


(def ^:private uuid-regex
  (let [x "[0-9a-fA-F]"] (re-pattern (str
    "^" x x x x x x x x "-" x x x x "-" x x x x "-" x x x x "-" x x x x x x x x x x x x "$"))))


(defn uuid?
  "(uuid? maybe-uuid)  =>  truthy-falsy
  Arguments and Values:
  maybe-uuid --- string or UUID-instance that may represent a conformant UUID.
  truthy-falsy --- Returns either the conforming UUID-string (truthy) or nil (falsy).
  Description:
  Predicate to test whether a string representation conforms to a
  \"xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx\" format where each x is a hexadecimal character.
  Input can be a maybe-uuid string or a cljs.core/UUID instance.
  Note that the current \"cljs.core/UUID.\" constructor does not check for any conformance.
  Examples:
  (uuid? \"NO-WAY\")  =>  nil
  (uuid? \"4d7332e7-e4c6-4ca5-af91-86336c825e25\")  => \"4d7332e7-e4c6-4ca5-af91-86336c825e25\"
  (uuid? (UUID. \"4d7332e7-e4c6-4ca5-af91-86336c825e25\"))  => \"4d7332e7-e4c6-4ca5-af91-86336c825e25\"
  (uuid? (UUID. \"YES-WAY\"))  => nil"
  [maybe-uuid]
  (let [maybe-uuid-str (cond
                         (= (type maybe-uuid) cljs.core/UUID) (to-str maybe-uuid)
                         (string? maybe-uuid) maybe-uuid
                         :true false)]
    (when maybe-uuid-str (re-find uuid-regex maybe-uuid-str))))


;; java equivalent "java.util.UUID/fromString" throws: IllegalArgumentException Invalid UUID string: ffa2a001-9eec-4224-a64d  java.util.UUID.fromString
;; to-uuid should probably throw an exception also instead of silently returning nil...

(defn to-uuid
  "(to-uuid maybe-uuid maybe-uuid)  =>  uuid-or-nil
  Arguments and Values:
  maybe-uuid --- string or UUID-instance that may represent a conformant UUID.
  uuid-or-nil --- Returns either a cljs.core/UUID instance or nil.
  Description:
  Returns a cljs.core/UUID instance for a conformant UUID-string representation, or nil.
  Input can be a string or a cljs.core/UUID instance.
  Note that if the input UUID-instance is not valid, nil is returned.
  Examples:
  (to-uuid \"NO-WAY\")  =>  nil
  (to-uuid \"4d7332e7-e4c6-4ca5-af91-86336c825e25\")  => #uuid \"4d7332e7-e4c6-4ca5-af91-86336c825e25\"
  (to-uuid (UUID. \"4d7332e7-e4c6-4ca5-af91-86336c825e25\"))  => #uuid \"4d7332e7-e4c6-4ca5-af91-86336c825e25\"
  (to-uuid (UUID. \"YES-WAY\"))  => nil"
  [maybe-uuid]
  (when-let [uuid (uuid? maybe-uuid)]
    (if (= (type maybe-uuid) cljs.core/UUID)
      maybe-uuid
      (UUID. uuid))))
