(ns om-justified.core
  (:require [om.core :as om :include-macros true]
            [om.dom :as dom :include-macros true]
            [goog.events :as events]
            [goog.events.EventType]))

(defn justify-sizes [sizes cont-width max-height spacing]
  ((fn step [value row row-width sizes]
     (if (seq sizes)
       (let [size-current    (first sizes)
             sizes-rest      (rest sizes)
             row-width'      (+ row-width (first size-current) spacing)
             row'            (conj row size-current)
             row-imgs-width' (apply + (map first row'))]
         (if (>= row-width' cont-width)
           (let [row-factor (/ row-imgs-width' (- cont-width (* (count row') spacing)))
                 row-height (/ max-height row-factor)
                 row-sizes  (map #(-> (assoc % 0 (-> (get % 0) (/ row-factor)))
                                      (assoc   1 (-> row-height)))
                                  row')
                 img-widths (map first row-sizes)
                 difference (-> (- (apply + img-widths) (->> (map js/Math.floor img-widths) (apply +))) js/Math.floor)
                 row-sizes' (map (fn [[w h] p]
                                   [(-> w js/Math.floor (+ p)) (js/Math.floor h)])
                                 row-sizes
                                 (concat (repeat difference 1) (repeat 0)))]
             (step (conj value row-sizes') []  0 sizes-rest))
           (if (not-empty sizes-rest)
             (step value row' row-width' (rest sizes))
             (conj value row'))))
       value))
   [] [] 0 sizes))

(defn sizes-to-images [rows images]
  ((fn step [value rows row-i images image-i]
     (let [row         (vec (first rows))
           row-size    (count row)
           row-images  (take row-size images)
           rows-rest   (rest rows)
           combined    (map-indexed #(-> (assoc %2 :just-width  (-> (get-in row [%1 0])))
                                         (assoc    :just-height (-> (get-in row [%1 1])))
                                         (assoc    :just-row    row-i))
                                    row-images)]
       (if (not-empty rows-rest)
         (step (conj value combined) rows-rest (inc row-i) (drop row-size images) (+ image-i row-size))
         (conj value combined))))
   [] rows 0 images 0))

(defn assoc-justified
  [images cont-width spacing]
  (if cont-width
    (time
      (let [sizes      (map #(vector (:width %) (:height %)) images)
            ;; - 1 off cont-width because .-clientWidth returns an integer so possibly rounded up
            just-sizes (justify-sizes sizes (- cont-width 1) (-> sizes first second) spacing)
            result     (sizes-to-images just-sizes images)]
        result))
    images))

(defn row-component
  [images owner {:keys [compo]}]
  (reify
    om/IRender
    (render [_]
      (apply dom/div nil
        (om/build-all compo images)))))

(defn justified-rows
  [{:keys [images] :as app} owner {:keys [compo spacing]}]
  (reify
    om/IDidMount
    (did-mount [_]
      (let [node (om/get-node owner "container")]
        (om/set-state! owner :cont-width (.-clientWidth node))
        (events/listen js/window goog.events.EventType.RESIZE
          (fn [event]
            (let [new-width (.-clientWidth node)]
              (when (not= (om/get-state owner :cont-width) new-width)
                (om/set-state! owner :cont-width new-width)))))))
    om/IRenderState
    (render-state [_ {container-width :cont-width}]
      (let [just-images (assoc-justified images container-width spacing)]
        (apply dom/div #js {:ref "container"}
          (om/build-all row-component just-images {:opts {:compo compo}}))))))

(defn justified
  [{:keys [images] :as app} owner {:keys [compo spacing]}]
  (reify
    om/IDidMount
    (did-mount [_]
      (let [node (om/get-node owner "container")]
        (om/set-state! owner :cont-width (.-clientWidth node))
        (events/listen js/window goog.events.EventType.RESIZE
          (fn [event]
            (let [new-width (.-clientWidth node)]
              (when (not= (om/get-state owner :cont-width) new-width)
                (om/set-state! owner :cont-width new-width)))))))
    om/IRenderState
    (render-state [_ {container-width :cont-width}]
      (let [just-images (assoc-justified images container-width spacing)]
        (apply dom/div #js {:ref "container"}
          (om/build-all compo (flatten just-images)))))))
