(ns clj-interest-graph.core
  (:require
    [clj-http.client :as http]))

(defn create-client
  ([] (create-client {}))
  ([options]
    (let [default-client
          {:base-url "https://interest-graph.getprismatic.com"
           :token nil}]
    (merge default-client options))))

(defn- api-get
  ([client endpoint] (get client endpoint {}))
  ([client endpoint params]
   (let [base-url (:base-url client)
         headers {"X-API-TOKEN" (:token client)}
         result (http/get (str base-url endpoint)
                   {:as :json
                    :content-type :json
                    :headers headers
                    :query-params params
                    :throw-entire-message? true})]
     (:body result))))

(defn- api-post
  "Makes POST request to the Interest Graph API."
  [client endpoint params]
  (let [base-url (:base-url client)
        headers {"X-API-TOKEN" (:token client)}
        result (http/post (str base-url endpoint)
                            {:as :json
                             :content-type :json
                             :form-params params
                             :headers headers
                             :throw-entire-message? true})]
      (:body result)))


;;; API Methods

(defn aspect
  "List all apects."
  [client]
  (api-get client "/aspect"))

(defn doc-search
  "Search for docs that satisfy the given query."
  [client query]
  (api-post client "/doc/search" {:query query}))

(defn ping
  "For health-check and testing."
  [client]
  (api-get client "/ping"))

(defn text-topic
  "Tag free text with topics."
  [client title body]
  (api-post client "/text/topic" {:title title :body body}))

(defn topic-recent-url
  "Deprecated - this endpoint will be removed. Use /doc/search instead."
  [client topic-id]
  (api-get client "/topic/recent-url" {:id topic-id}))

;; TODO: Implement limit.
(defn topic-search
  "Topics matching a query string."
  [client search-query]
  (api-get client "/topic/search" {:search-query search-query}))

(defn topic-topic
  "Topics similar to a given topic."
  [client topic-id]
  (api-get client "/topic/search" {:id topic-id}))

(defn url
  "All annotations (e.g. topics and aspects) for a given url"
  [client url]
  (api-post client "/url" {:url url}))

(defn url-aspect
  "Lookup aspects for a given url"
  [client url]
  (api-post client "/url/aspect" {:url url}))

(defn url-topic
  "Tag a url with topics"
  [client url]
  (api-post client "/url/topic" {:url url}))
