(ns wise.auth.middleware
  (:require [clojure.string :as cstr]
            [compojure.api.sweet :refer :all]
            [compojure.api.meta :refer [restructure-param src-coerce!]]
            [ring.util.http-response :refer [unauthorized]]
            [clj-time.core :as time]
            [buddy.auth :refer [authenticated?]]
            [buddy.auth.backends.token :refer [jws-backend]]
            [buddy.auth.middleware :refer [wrap-authentication]]
            [buddy.sign.jwt :as jwt]))

;; 使用前需要实现 get-auth 方法, xx-fn获取用户的所有拥有权限的uri地址集合，入参m为{:id 用户id}
;; [wise.auth.middleware :refer [get-auth]]
;; (defmethod get-auth :default [m] (xx-fn m))

(defmulti get-auth identity)

(def ^:private secret "wisdragon")

(def ^:private auth-backend (jws-backend {:secret secret}))

(defn sign-token [claims]
  (-> {:user claims}
      (merge {:exp (time/plus (time/now) (time/seconds 3600))})
      (jwt/sign secret)))

(defn unsign-token [token]
  (-> token
      (jwt/unsign secret)))

(defn limit-auth? [u id]
  (let [auths (:auths u)]
    (some #(= id %) auths)))

(defn own-auth? [u id]
  (if (= 100 (-> u :id))
    true
    (limit-auth? u id)))

(defn- pattern-urls [urls]
  (map #(re-pattern (str "^" % ".*")) urls))

(defn- permission? [request]
  (let [request-url (:uri request)
        user-id (-> request :identity :user :id)
        auth-urls (get-auth {:id user-id})
        auth-urls (remove cstr/blank? auth-urls)
        auth-urls (pattern-urls auth-urls)]
    (if (empty? auth-urls)
      false
      (boolean (some #(re-matches % request-url) auth-urls)))))

(defn authenticated-mw
  "Middleware used in routes that require authentication. If request is not
   authenticated a 401 not authorized response will be returned"
  [handler]
  (fn [request]
    (if (authenticated? request)
      (if (permission? request)
        (handler request)
        (unauthorized "没有权限！"))
      (unauthorized "没有登陆！"))))

(defn token-auth-mw
  "Middleware used on routes requiring token authentication"
  [handler]
  (-> handler
      (wrap-authentication auth-backend)))

(defmethod restructure-param :current-user
  [_ binding acc]
  (update-in acc [:letks] into [binding `(:user (:identity ~'+compojure-api-request+))]))
