(ns wiseloong.router
  (:require [reagent.core :as r]
            [bidi.router :as br]))

(def ^:private !location (r/atom {:route-params nil :handler nil}))

(defn router-page
  ([] (router-page !location))
  ([!loc]
   (fn []
     (let [handler (:handler @!loc)
           route-params (r/cursor !loc [:route-params])]
       [handler route-params]))))

(defn router!
  "Starts up a Bidi router based on Google Closure's 'History'
  Example usage:
   (ns comp.core
     (:require [comp.router :as cr]))

   (defn home-page []
     [:div \"home-page\"
       [:a {:href \"#/about\"} \"about-page\"]])

   (defn a-page [item-id]
     (println @item-id)
     [:div \"a-page\"
       [:a {:href \"#/\"} \"home-page\"]])
   ...

   (def app-routes
     [\"/\" {\"\"        home-page
             \"a-items\" {\"\"                  a-pages
                          [\"/item-\" :item-id] a-page}
             \"about\"   about-page
             true        error-page}])

   (defn main []
     [:div [cr/router-page]]))

   (defn init! []
     (cr/router! app-routes)
     (r/render [main] (.getElementById js/document \"app\"))

   (init!)"
  ([routes] (router! routes !location))
  ([routes !loc] (br/start-router! routes
                                   {:on-navigate (fn [location] (reset! !loc location))})))

(defprotocol Router
  (get-location! [_]))

(defn start-router!
  "这种方式比上面router!更灵活，可以扩展Router方法，实现更多功能。
  Starts up a Bidi router based on Google Closure's 'History'
  Example usage:
   (ns comp.core
     (:require [comp.router :as cr]))

   (defn home-page []
     [:div \"home-page\"
       [:a {:href \"#/about\"} \"about\"]])

   (defn a-page [item-id]
     (println @item-id)
       [:div \"a-page\"
         [:a {:href \"#/\"} \"home-page\"]])
   ...

   (def app-routes
     [\"/\" {\"\"        home-page
             \"a-items\" {\"\"                  a-pages
                          [\"/item-\" :item-id] a-page}
             \"about\"   about-page
             true        error-page}])

   (defn main [r]
     [:div [cr/get-location! r]]))

   (defn init! []
     (let [router (cr/start-router! app-routes)]
       (r/render [main router] (.getElementById js/document \"app\")))

   (init!)"
  [routes]
  (let [!loc (r/atom {:route-params nil :handler nil})]
    (router! routes !loc)
    (reify Router
      (get-location! [_]
        (router-page !loc)))))