/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.core;

import com.google.bitcoin.core.Address;
import com.google.bitcoin.core.DumpedPrivateKey;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.UnsafeByteArrayOutputStream;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.crypto.EncryptedPrivateKey;
import com.google.bitcoin.crypto.KeyCrypter;
import com.google.bitcoin.crypto.KeyCrypterException;
import com.google.common.base.Preconditions;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Serializable;
import java.math.BigInteger;
import java.nio.charset.Charset;
import java.security.SecureRandom;
import java.security.SignatureException;
import java.util.Arrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spongycastle.asn1.ASN1Encodable;
import org.spongycastle.asn1.ASN1InputStream;
import org.spongycastle.asn1.ASN1Integer;
import org.spongycastle.asn1.ASN1OctetString;
import org.spongycastle.asn1.DERBitString;
import org.spongycastle.asn1.DERInteger;
import org.spongycastle.asn1.DEROctetString;
import org.spongycastle.asn1.DERSequenceGenerator;
import org.spongycastle.asn1.DERTaggedObject;
import org.spongycastle.asn1.DLSequence;
import org.spongycastle.asn1.sec.SECNamedCurves;
import org.spongycastle.asn1.x9.X9ECParameters;
import org.spongycastle.crypto.AsymmetricCipherKeyPair;
import org.spongycastle.crypto.CipherParameters;
import org.spongycastle.crypto.KeyGenerationParameters;
import org.spongycastle.crypto.generators.ECKeyPairGenerator;
import org.spongycastle.crypto.params.ECDomainParameters;
import org.spongycastle.crypto.params.ECKeyGenerationParameters;
import org.spongycastle.crypto.params.ECPrivateKeyParameters;
import org.spongycastle.crypto.params.ECPublicKeyParameters;
import org.spongycastle.crypto.params.KeyParameter;
import org.spongycastle.crypto.signers.ECDSASigner;
import org.spongycastle.math.ec.ECCurve;
import org.spongycastle.math.ec.ECFieldElement;
import org.spongycastle.math.ec.ECPoint;
import org.spongycastle.util.encoders.Base64;

public class ECKey
implements Serializable {
    private static final Logger log = LoggerFactory.getLogger(ECKey.class);
    private static final ECDomainParameters ecParams;
    private static final SecureRandom secureRandom;
    private static final long serialVersionUID = -728224901792295832L;
    private BigInteger priv;
    private byte[] pub;
    private long creationTimeSeconds;
    private transient KeyCrypter keyCrypter;
    private EncryptedPrivateKey encryptedPrivateKey;
    private transient byte[] pubKeyHash;

    public ECKey() {
        ECKeyPairGenerator generator = new ECKeyPairGenerator();
        ECKeyGenerationParameters keygenParams = new ECKeyGenerationParameters(ecParams, secureRandom);
        generator.init((KeyGenerationParameters)keygenParams);
        AsymmetricCipherKeyPair keypair = generator.generateKeyPair();
        ECPrivateKeyParameters privParams = (ECPrivateKeyParameters)keypair.getPrivate();
        ECPublicKeyParameters pubParams = (ECPublicKeyParameters)keypair.getPublic();
        this.priv = privParams.getD();
        ECPoint uncompressed = pubParams.getQ();
        ECPoint compressed = ECKey.compressPoint(uncompressed);
        this.pub = compressed.getEncoded();
        this.creationTimeSeconds = Utils.now().getTime() / 1000L;
    }

    private static ECPoint compressPoint(ECPoint uncompressed) {
        return new ECPoint.Fp(ecParams.getCurve(), uncompressed.getX(), uncompressed.getY(), true);
    }

    public static ECKey fromASN1(byte[] asn1privkey) {
        return new ECKey(ECKey.extractPrivateKeyFromASN1(asn1privkey));
    }

    public ECKey(BigInteger privKey) {
        this(privKey, (byte[])null);
    }

    public ECKey(BigInteger privKey, BigInteger pubKey) {
        this(privKey, Utils.bigIntegerToBytes(pubKey, 65));
    }

    public ECKey(byte[] privKeyBytes, byte[] pubKey) {
        this(privKeyBytes == null ? null : new BigInteger(1, privKeyBytes), pubKey);
    }

    public ECKey(EncryptedPrivateKey encryptedPrivateKey, byte[] pubKey, KeyCrypter keyCrypter) {
        this((byte[])null, pubKey);
        this.keyCrypter = (KeyCrypter)Preconditions.checkNotNull((Object)keyCrypter);
        this.encryptedPrivateKey = encryptedPrivateKey;
    }

    public ECKey(BigInteger privKey, byte[] pubKey, boolean compressed) {
        this.priv = privKey;
        this.pub = null;
        if (pubKey == null && privKey != null) {
            this.pub = ECKey.publicKeyFromPrivate(privKey, compressed);
        } else if (pubKey != null) {
            this.pub = pubKey;
        }
    }

    private ECKey(BigInteger privKey, byte[] pubKey) {
        this(privKey, pubKey, false);
    }

    public byte[] toASN1() {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream(400);
            DERSequenceGenerator seq = new DERSequenceGenerator((OutputStream)baos);
            seq.addObject((ASN1Encodable)new ASN1Integer(1));
            seq.addObject((ASN1Encodable)new DEROctetString(this.priv.toByteArray()));
            seq.addObject((ASN1Encodable)new DERTaggedObject(0, (ASN1Encodable)SECNamedCurves.getByName((String)"secp256k1").toASN1Primitive()));
            seq.addObject((ASN1Encodable)new DERTaggedObject(1, (ASN1Encodable)new DERBitString(this.getPubKey())));
            seq.close();
            return baos.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] publicKeyFromPrivate(BigInteger privKey, boolean compressed) {
        ECPoint point = ecParams.getG().multiply(privKey);
        if (compressed) {
            point = ECKey.compressPoint(point);
        }
        return point.getEncoded();
    }

    public byte[] getPubKeyHash() {
        if (this.pubKeyHash == null) {
            this.pubKeyHash = Utils.sha256hash160(this.pub);
        }
        return this.pubKeyHash;
    }

    public byte[] getPubKey() {
        return this.pub;
    }

    public boolean isCompressed() {
        return this.pub.length == 33;
    }

    public String toString() {
        StringBuilder b = new StringBuilder();
        b.append("pub:").append(Utils.bytesToHexString(this.pub));
        if (this.creationTimeSeconds != 0L) {
            b.append(" timestamp:").append(this.creationTimeSeconds);
        }
        if (this.isEncrypted()) {
            b.append(" encrypted");
        }
        return b.toString();
    }

    public String toStringWithPrivate() {
        StringBuilder b = new StringBuilder();
        b.append(this.toString());
        if (this.priv != null) {
            b.append(" priv:").append(Utils.bytesToHexString(this.priv.toByteArray()));
        }
        return b.toString();
    }

    public Address toAddress(NetworkParameters params) {
        byte[] hash160 = Utils.sha256hash160(this.pub);
        return new Address(params, hash160);
    }

    public void clearPrivateKey() {
        this.priv = BigInteger.ZERO;
        if (this.encryptedPrivateKey != null) {
            this.encryptedPrivateKey.clear();
        }
    }

    public ECDSASignature sign(Sha256Hash input) throws KeyCrypterException {
        return this.sign(input, null);
    }

    public ECDSASignature sign(Sha256Hash input, KeyParameter aesKey) throws KeyCrypterException {
        BigInteger privateKeyForSigning;
        if (this.isEncrypted()) {
            if (aesKey == null) {
                throw new KeyCrypterException("This ECKey is encrypted but no decryption key has been supplied.");
            }
            if (this.keyCrypter == null) {
                throw new KeyCrypterException("There is no KeyCrypter to decrypt the private key for signing.");
            }
            privateKeyForSigning = new BigInteger(1, this.keyCrypter.decrypt(this.encryptedPrivateKey, aesKey));
            if (!Arrays.equals(this.pub, ECKey.publicKeyFromPrivate(privateKeyForSigning, this.isCompressed()))) {
                throw new KeyCrypterException("Could not decrypt bytes");
            }
        } else {
            if (this.priv == null) {
                throw new KeyCrypterException("This ECKey does not have the private key necessary for signing.");
            }
            privateKeyForSigning = this.priv;
        }
        ECDSASigner signer = new ECDSASigner();
        ECPrivateKeyParameters privKey = new ECPrivateKeyParameters(privateKeyForSigning, ecParams);
        signer.init(true, (CipherParameters)privKey);
        BigInteger[] sigs = signer.generateSignature(input.getBytes());
        return new ECDSASignature(sigs[0], sigs[1]);
    }

    public static boolean verify(byte[] data, byte[] signature, byte[] pub) {
        ECDSASigner signer = new ECDSASigner();
        ECPublicKeyParameters params = new ECPublicKeyParameters(ecParams.getCurve().decodePoint(pub), ecParams);
        signer.init(false, (CipherParameters)params);
        try {
            ASN1InputStream decoder = new ASN1InputStream(signature);
            DLSequence seq = (DLSequence)decoder.readObject();
            DERInteger r = (DERInteger)seq.getObjectAt(0);
            DERInteger s = (DERInteger)seq.getObjectAt(1);
            decoder.close();
            try {
                return signer.verifySignature(data, r.getPositiveValue(), s.getPositiveValue());
            }
            catch (NullPointerException e) {
                log.error("Caught NPE inside bouncy castle");
                e.printStackTrace();
                return false;
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public boolean verify(byte[] data, byte[] signature) {
        return ECKey.verify(data, signature, this.pub);
    }

    private static BigInteger extractPrivateKeyFromASN1(byte[] asn1privkey) {
        try {
            ASN1InputStream decoder = new ASN1InputStream(asn1privkey);
            DLSequence seq = (DLSequence)decoder.readObject();
            Preconditions.checkArgument((seq.size() == 4 ? 1 : 0) != 0, (Object)"Input does not appear to be an ASN.1 OpenSSL EC private key");
            Preconditions.checkArgument((boolean)((DERInteger)seq.getObjectAt(0)).getValue().equals(BigInteger.ONE), (Object)"Input is of wrong version");
            ASN1Encodable obj = seq.getObjectAt(1);
            byte[] bits = ((ASN1OctetString)obj).getOctets();
            decoder.close();
            return new BigInteger(1, bits);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public String signMessage(String message) throws KeyCrypterException {
        return this.signMessage(message, null);
    }

    public String signMessage(String message, KeyParameter aesKey) throws KeyCrypterException {
        if (this.priv == null) {
            throw new IllegalStateException("This ECKey does not have the private key necessary for signing.");
        }
        byte[] data = Utils.formatMessageForSigning(message);
        Sha256Hash hash = Sha256Hash.createDouble(data);
        ECDSASignature sig = this.sign(hash, aesKey);
        int recId = -1;
        for (int i = 0; i < 4; ++i) {
            ECKey k = ECKey.recoverFromSignature(i, sig, hash, this.isCompressed());
            if (k == null || !Arrays.equals(k.pub, this.pub)) continue;
            recId = i;
            break;
        }
        if (recId == -1) {
            throw new RuntimeException("Could not construct a recoverable key. This should never happen.");
        }
        int headerByte = recId + 27 + (this.isCompressed() ? 4 : 0);
        byte[] sigData = new byte[65];
        sigData[0] = (byte)headerByte;
        System.arraycopy(Utils.bigIntegerToBytes(sig.r, 32), 0, sigData, 1, 32);
        System.arraycopy(Utils.bigIntegerToBytes(sig.s, 32), 0, sigData, 33, 32);
        return new String(Base64.encode((byte[])sigData), Charset.forName("UTF-8"));
    }

    public static ECKey signedMessageToKey(String message, String signatureBase64) throws SignatureException {
        int recId;
        ECKey key;
        byte[] signatureEncoded;
        try {
            signatureEncoded = Base64.decode((String)signatureBase64);
        }
        catch (RuntimeException e) {
            throw new SignatureException("Could not decode base64", e);
        }
        if (signatureEncoded.length < 65) {
            throw new SignatureException("Signature truncated, expected 65 bytes and got " + signatureEncoded.length);
        }
        int header = signatureEncoded[0] & 0xFF;
        if (header < 27 || header > 34) {
            throw new SignatureException("Header byte out of range: " + header);
        }
        BigInteger r = new BigInteger(1, Arrays.copyOfRange(signatureEncoded, 1, 33));
        BigInteger s = new BigInteger(1, Arrays.copyOfRange(signatureEncoded, 33, 65));
        ECDSASignature sig = new ECDSASignature(r, s);
        byte[] messageBytes = Utils.formatMessageForSigning(message);
        Sha256Hash messageHash = Sha256Hash.createDouble(messageBytes);
        boolean compressed = false;
        if (header >= 31) {
            compressed = true;
            header -= 4;
        }
        if ((key = ECKey.recoverFromSignature(recId = header - 27, sig, messageHash, compressed)) == null) {
            throw new SignatureException("Could not recover public key from signature");
        }
        return key;
    }

    public void verifyMessage(String message, String signatureBase64) throws SignatureException {
        ECKey key = ECKey.signedMessageToKey(message, signatureBase64);
        if (!Arrays.equals(key.getPubKey(), this.pub)) {
            throw new SignatureException("Signature did not match for message");
        }
    }

    public static ECKey recoverFromSignature(int recId, ECDSASignature sig, Sha256Hash message, boolean compressed) {
        Preconditions.checkArgument((recId >= 0 ? 1 : 0) != 0, (Object)"recId must be positive");
        Preconditions.checkArgument((sig.r.compareTo(BigInteger.ZERO) >= 0 ? 1 : 0) != 0, (Object)"r must be positive");
        Preconditions.checkArgument((sig.s.compareTo(BigInteger.ZERO) >= 0 ? 1 : 0) != 0, (Object)"s must be positive");
        Preconditions.checkNotNull((Object)message);
        BigInteger n = ecParams.getN();
        BigInteger i = BigInteger.valueOf((long)recId / 2L);
        BigInteger x = sig.r.add(i.multiply(n));
        ECCurve.Fp curve = (ECCurve.Fp)ecParams.getCurve();
        BigInteger prime = curve.getQ();
        if (x.compareTo(prime) >= 0) {
            return null;
        }
        ECPoint R = ECKey.decompressKey(x, (recId & 1) == 1);
        if (!R.multiply(n).isInfinity()) {
            return null;
        }
        BigInteger e = message.toBigInteger();
        BigInteger eInv = BigInteger.ZERO.subtract(e).mod(n);
        BigInteger rInv = sig.r.modInverse(n);
        BigInteger srInv = rInv.multiply(sig.s).mod(n);
        BigInteger eInvrInv = rInv.multiply(eInv).mod(n);
        ECPoint p1 = ecParams.getG().multiply(eInvrInv);
        ECPoint p2 = R.multiply(srInv);
        ECPoint.Fp q = (ECPoint.Fp)p2.add(p1);
        if (compressed) {
            q = new ECPoint.Fp((ECCurve)curve, q.getX(), q.getY(), true);
        }
        return new ECKey((byte[])null, q.getEncoded());
    }

    private static ECPoint decompressKey(BigInteger xBN, boolean yBit) {
        ECCurve.Fp curve = (ECCurve.Fp)ecParams.getCurve();
        ECFieldElement.Fp x = new ECFieldElement.Fp(curve.getQ(), xBN);
        ECFieldElement alpha = x.multiply(x.square().add(curve.getA())).add(curve.getB());
        ECFieldElement beta = alpha.sqrt();
        if (beta == null) {
            throw new IllegalArgumentException("Invalid point compression");
        }
        if (beta.toBigInteger().testBit(0) == yBit) {
            return new ECPoint.Fp((ECCurve)curve, (ECFieldElement)x, beta, true);
        }
        ECFieldElement.Fp y = new ECFieldElement.Fp(curve.getQ(), curve.getQ().subtract(beta.toBigInteger()));
        return new ECPoint.Fp((ECCurve)curve, (ECFieldElement)x, (ECFieldElement)y, true);
    }

    public byte[] getPrivKeyBytes() {
        return Utils.bigIntegerToBytes(this.priv, 32);
    }

    public DumpedPrivateKey getPrivateKeyEncoded(NetworkParameters params) {
        return new DumpedPrivateKey(params, this.getPrivKeyBytes(), this.isCompressed());
    }

    public long getCreationTimeSeconds() {
        return this.creationTimeSeconds;
    }

    public void setCreationTimeSeconds(long newCreationTimeSeconds) {
        if (newCreationTimeSeconds < 0L) {
            throw new IllegalArgumentException("Cannot set creation time to negative value: " + newCreationTimeSeconds);
        }
        this.creationTimeSeconds = newCreationTimeSeconds;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ECKey ecKey = (ECKey)o;
        return Arrays.equals(this.pub, ecKey.pub);
    }

    public int hashCode() {
        return this.pub[0] & 0xFF | (this.pub[1] & 0xFF) << 8 | (this.pub[2] & 0xFF) << 16 | (this.pub[3] & 0xFF) << 24;
    }

    public ECKey encrypt(KeyCrypter keyCrypter, KeyParameter aesKey) throws KeyCrypterException {
        Preconditions.checkNotNull((Object)keyCrypter);
        EncryptedPrivateKey encryptedPrivateKey = keyCrypter.encrypt(this.getPrivKeyBytes(), aesKey);
        return new ECKey(encryptedPrivateKey, this.getPubKey(), keyCrypter);
    }

    public ECKey decrypt(KeyCrypter keyCrypter, KeyParameter aesKey) throws KeyCrypterException {
        Preconditions.checkNotNull((Object)keyCrypter);
        if (this.keyCrypter != null && !this.keyCrypter.equals(keyCrypter)) {
            throw new KeyCrypterException("The keyCrypter being used to decrypt the key is different to the one that was used to encrypt it");
        }
        byte[] unencryptedPrivateKey = keyCrypter.decrypt(this.encryptedPrivateKey, aesKey);
        ECKey key = new ECKey(new BigInteger(1, unencryptedPrivateKey), null, this.isCompressed());
        if (!Arrays.equals(key.getPubKey(), this.getPubKey())) {
            throw new KeyCrypterException("Provided AES key is wrong");
        }
        return key;
    }

    public static boolean encryptionIsReversible(ECKey originalKey, ECKey encryptedKey, KeyCrypter keyCrypter, KeyParameter aesKey) {
        String genericErrorText = "The check that encryption could be reversed failed for key " + originalKey.toString() + ". ";
        try {
            ECKey rebornUnencryptedKey = encryptedKey.decrypt(keyCrypter, aesKey);
            if (rebornUnencryptedKey == null) {
                log.error(genericErrorText + "The test decrypted key was missing.");
                return false;
            }
            byte[] originalPrivateKeyBytes = originalKey.getPrivKeyBytes();
            if (originalPrivateKeyBytes != null) {
                if (rebornUnencryptedKey.getPrivKeyBytes() == null) {
                    log.error(genericErrorText + "The test decrypted key was missing.");
                    return false;
                }
                if (originalPrivateKeyBytes.length != rebornUnencryptedKey.getPrivKeyBytes().length) {
                    log.error(genericErrorText + "The test decrypted private key was a different length to the original.");
                    return false;
                }
                for (int i = 0; i < originalPrivateKeyBytes.length; ++i) {
                    if (originalPrivateKeyBytes[i] == rebornUnencryptedKey.getPrivKeyBytes()[i]) continue;
                    log.error(genericErrorText + "Byte " + i + " of the private key did not match the original.");
                    return false;
                }
            }
        }
        catch (KeyCrypterException kce) {
            log.error(kce.getMessage());
            return false;
        }
        return true;
    }

    public boolean isEncrypted() {
        return this.keyCrypter != null && this.encryptedPrivateKey != null && this.encryptedPrivateKey.getEncryptedBytes() != null && this.encryptedPrivateKey.getEncryptedBytes().length > 0;
    }

    public EncryptedPrivateKey getEncryptedPrivateKey() {
        if (this.encryptedPrivateKey == null) {
            return null;
        }
        return this.encryptedPrivateKey.clone();
    }

    public KeyCrypter getKeyCrypter() {
        return this.keyCrypter;
    }

    static {
        X9ECParameters params = SECNamedCurves.getByName((String)"secp256k1");
        ecParams = new ECDomainParameters(params.getCurve(), params.getG(), params.getN(), params.getH());
        secureRandom = new SecureRandom();
    }

    public static class ECDSASignature {
        public BigInteger r;
        public BigInteger s;

        public ECDSASignature(BigInteger r, BigInteger s) {
            this.r = r;
            this.s = s;
        }

        public byte[] encodeToDER() {
            try {
                UnsafeByteArrayOutputStream bos = new UnsafeByteArrayOutputStream(72);
                DERSequenceGenerator seq = new DERSequenceGenerator((OutputStream)bos);
                seq.addObject((ASN1Encodable)new DERInteger(this.r));
                seq.addObject((ASN1Encodable)new DERInteger(this.s));
                seq.close();
                return ((ByteArrayOutputStream)bos).toByteArray();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }
}

