/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.core;

import com.google.bitcoin.core.AbstractBlockChain;
import com.google.bitcoin.core.Address;
import com.google.bitcoin.core.BlockChainListener;
import com.google.bitcoin.core.BloomFilter;
import com.google.bitcoin.core.ECKey;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.Peer;
import com.google.bitcoin.core.PeerGroup;
import com.google.bitcoin.core.ScriptException;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.StoredBlock;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.TransactionConfidence;
import com.google.bitcoin.core.TransactionInput;
import com.google.bitcoin.core.TransactionOutPoint;
import com.google.bitcoin.core.TransactionOutput;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.core.VerificationException;
import com.google.bitcoin.core.WalletEventListener;
import com.google.bitcoin.core.WalletTransaction;
import com.google.bitcoin.crypto.KeyCrypter;
import com.google.bitcoin.crypto.KeyCrypterException;
import com.google.bitcoin.crypto.KeyCrypterScrypt;
import com.google.bitcoin.store.WalletProtobufSerializer;
import com.google.bitcoin.utils.Locks;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ListenableFuture;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.bitcoinj.wallet.Protos;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spongycastle.crypto.params.KeyParameter;

public class Wallet
implements Serializable,
BlockChainListener {
    private static final Logger log = LoggerFactory.getLogger(Wallet.class);
    private static final long serialVersionUID = 2L;
    protected final ReentrantLock lock = Locks.lock("wallet");
    final Map<Sha256Hash, Transaction> pending;
    final Map<Sha256Hash, Transaction> unspent;
    final Map<Sha256Hash, Transaction> spent;
    final Map<Sha256Hash, Transaction> inactive;
    final Map<Sha256Hash, Transaction> dead;
    public ArrayList<ECKey> keychain;
    private final NetworkParameters params;
    private Sha256Hash lastBlockSeenHash;
    private int lastBlockSeenHeight = -1;
    private transient CopyOnWriteArrayList<WalletEventListener> eventListeners;
    private transient File autosaveToFile;
    private transient boolean dirty;
    private transient AutosaveEventListener autosaveEventListener;
    private transient long autosaveDelayMs;
    private transient TransactionConfidence.Listener txConfidenceListener;
    private transient HashSet<Sha256Hash> ignoreNextNewBlock;
    private boolean acceptTimeLockedTransactions;
    private transient CoinSelector coinSelector = new DefaultCoinSelector();
    private KeyCrypter keyCrypter;
    private int version;
    private String description;
    private int onWalletChangedSuppressions = 0;

    public Wallet(NetworkParameters params) {
        this(params, null);
    }

    public Wallet(NetworkParameters params, KeyCrypter keyCrypter) {
        this.keyCrypter = keyCrypter;
        this.params = (NetworkParameters)Preconditions.checkNotNull((Object)params);
        this.keychain = new ArrayList();
        this.unspent = new HashMap<Sha256Hash, Transaction>();
        this.spent = new HashMap<Sha256Hash, Transaction>();
        this.inactive = new HashMap<Sha256Hash, Transaction>();
        this.pending = new HashMap<Sha256Hash, Transaction>();
        this.dead = new HashMap<Sha256Hash, Transaction>();
        this.eventListeners = new CopyOnWriteArrayList();
        this.createTransientState();
    }

    private void createTransientState() {
        this.ignoreNextNewBlock = new HashSet();
        this.txConfidenceListener = new TransactionConfidence.Listener(){

            @Override
            public void onConfidenceChanged(Transaction tx) {
                Wallet.this.lock.lock();
                Wallet.this.invokeOnTransactionConfidenceChanged(tx);
                Wallet.this.invokeOnWalletChanged();
                Wallet.this.lock.unlock();
            }
        };
        this.acceptTimeLockedTransactions = false;
    }

    public NetworkParameters getNetworkParameters() {
        return this.params;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Iterable<ECKey> getKeys() {
        this.lock.lock();
        try {
            ArrayList<ECKey> arrayList = new ArrayList<ECKey>(this.keychain);
            return arrayList;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getKeychainSize() {
        this.lock.lock();
        try {
            int n = this.keychain.size();
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void saveToFile(File temp, File destFile) throws IOException {
        FileOutputStream stream = null;
        try {
            stream = new FileOutputStream(temp);
            this.saveToFileStream(stream);
            stream.flush();
            stream.getFD().sync();
            stream.close();
            stream = null;
            if (Utils.isWindows()) {
                File canonical = destFile.getCanonicalFile();
                canonical.delete();
                if (temp.renameTo(canonical)) {
                    return;
                }
                throw new IOException("Failed to rename " + temp + " to " + canonical);
            }
            if (!temp.renameTo(destFile)) {
                throw new IOException("Failed to rename " + temp + " to " + destFile);
            }
            this.lock.lock();
            try {
                if (destFile.equals(this.autosaveToFile)) {
                    this.dirty = false;
                }
            }
            finally {
                this.lock.unlock();
            }
        }
        finally {
            if (stream != null) {
                stream.close();
            }
            if (temp.delete()) {
                log.warn("Deleted temp file after failed save.");
            }
        }
    }

    public void saveToFile(File f) throws IOException {
        File directory = f.getAbsoluteFile().getParentFile();
        File temp = File.createTempFile("wallet", null, directory);
        this.saveToFile(temp, f);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setAcceptTimeLockedTransactions(boolean acceptTimeLockedTransactions) {
        this.lock.lock();
        try {
            this.acceptTimeLockedTransactions = acceptTimeLockedTransactions;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean doesAcceptTimeLockedTransactions() {
        this.lock.lock();
        try {
            boolean bl = this.acceptTimeLockedTransactions;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    private boolean autoSave() {
        this.lock.lock();
        Sha256Hash lastBlockSeenHash = this.lastBlockSeenHash;
        AutosaveEventListener autosaveEventListener = this.autosaveEventListener;
        File autosaveToFile = this.autosaveToFile;
        this.lock.unlock();
        try {
            log.info("Auto-saving wallet, last seen block is {}", (Object)lastBlockSeenHash);
            File directory = autosaveToFile.getAbsoluteFile().getParentFile();
            File temp = File.createTempFile("wallet", null, directory);
            if (autosaveEventListener != null) {
                autosaveEventListener.onBeforeAutoSave(temp);
            }
            this.saveToFile(temp, autosaveToFile);
            if (autosaveEventListener != null) {
                autosaveEventListener.onAfterAutoSave(autosaveToFile);
            }
        }
        catch (Exception e) {
            if (autosaveEventListener != null && autosaveEventListener.caughtException(e)) {
                return true;
            }
            throw new RuntimeException(e);
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void autosaveToFile(File f, long delayTime, TimeUnit timeUnit, AutosaveEventListener eventListener) {
        this.lock.lock();
        try {
            Preconditions.checkArgument((delayTime >= 0L ? 1 : 0) != 0);
            this.autosaveToFile = (File)Preconditions.checkNotNull((Object)f);
            if (delayTime > 0L) {
                this.autosaveEventListener = eventListener;
                this.autosaveDelayMs = TimeUnit.MILLISECONDS.convert(delayTime, timeUnit);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void queueAutoSave() {
        block9: {
            this.lock.lock();
            try {
                if (this.autosaveToFile == null) {
                    return;
                }
                if (this.autosaveDelayMs == 0L) {
                    try {
                        this.saveToFile(this.autosaveToFile);
                        break block9;
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
                if (!this.dirty) {
                    this.dirty = true;
                    AutosaveThread.registerForSave(this, this.autosaveDelayMs);
                }
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void saveToFileStream(OutputStream f) throws IOException {
        this.lock.lock();
        try {
            new WalletProtobufSerializer().writeWallet(this, f);
        }
        finally {
            this.lock.unlock();
        }
    }

    public NetworkParameters getParams() {
        return this.params;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Wallet loadFromFile(File f) throws IOException {
        FileInputStream stream = new FileInputStream(f);
        try {
            Wallet wallet = Wallet.loadFromFileStream(stream);
            return wallet;
        }
        finally {
            stream.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isConsistent() {
        this.lock.lock();
        try {
            int size2;
            boolean success = true;
            HashSet<Transaction> pendingInactive = new HashSet<Transaction>();
            pendingInactive.addAll(this.pending.values());
            pendingInactive.addAll(this.inactive.values());
            Set<Transaction> transactions = this.getTransactions(true, true);
            HashSet<Sha256Hash> hashes = new HashSet<Sha256Hash>();
            for (Transaction tx : transactions) {
                hashes.add(tx.getHash());
            }
            int size1 = transactions.size();
            if (size1 != hashes.size()) {
                log.error("Two transactions with same hash");
                success = false;
            }
            if (size1 != (size2 = this.unspent.size() + this.spent.size() + pendingInactive.size() + this.dead.size())) {
                log.error("Inconsistent wallet sizes: {} {}", (Object)size1, (Object)size2);
                success = false;
            }
            for (Transaction tx : this.unspent.values()) {
                if (tx.isConsistent(this, false)) continue;
                success = false;
                log.error("Inconsistent unspent tx {}", (Object)tx.getHashAsString());
            }
            for (Transaction tx : this.spent.values()) {
                if (tx.isConsistent(this, true)) continue;
                success = false;
                log.error("Inconsistent spent tx {}", (Object)tx.getHashAsString());
            }
            if (!success) {
                log.error(this.toString());
            }
            boolean bl = success;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    public static Wallet loadFromFileStream(InputStream stream) throws IOException {
        Wallet wallet;
        stream = new BufferedInputStream(stream);
        stream.mark(100);
        boolean serialization = stream.read() == 172 && stream.read() == 237;
        stream.reset();
        if (serialization) {
            ObjectInputStream ois = null;
            try {
                ois = new ObjectInputStream(stream);
                wallet = (Wallet)ois.readObject();
            }
            catch (ClassNotFoundException e) {
                throw new RuntimeException(e);
            }
            finally {
                if (ois != null) {
                    ois.close();
                }
            }
        } else {
            wallet = new WalletProtobufSerializer().readWallet(stream);
        }
        if (!wallet.isConsistent()) {
            log.error("Loaded an inconsistent wallet");
        }
        return wallet;
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        this.createTransientState();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void notifyTransactionIsInBlock(Sha256Hash txHash, StoredBlock block, AbstractBlockChain.NewBlockType blockType) throws VerificationException {
        this.lock.lock();
        try {
            Transaction tx = this.pending.get(txHash);
            if (tx == null) {
                return;
            }
            this.receive(tx, block, blockType, false);
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void receivePending(Transaction tx, List<Transaction> dependencies) throws VerificationException {
        this.lock.lock();
        try {
            TransactionConfidence.ConfidenceType currentConfidence;
            if (!this.isPendingTransactionRelevant(tx)) {
                return;
            }
            AnalysisResult analysis = Wallet.analyzeTransactionAndDependencies(tx, dependencies);
            if (analysis.timeLocked != null && !this.doesAcceptTimeLockedTransactions()) {
                log.warn("Transaction {}, dependency of {} has a time lock value of {}", new Object[]{analysis.timeLocked.getHashAsString(), tx.getHashAsString(), analysis.timeLocked.getLockTime()});
                return;
            }
            BigInteger valueSentToMe = tx.getValueSentToMe(this);
            BigInteger valueSentFromMe = tx.getValueSentFromMe(this);
            if (log.isInfoEnabled()) {
                log.info(String.format("Received a pending transaction %s that spends %s BTC from our own wallet, and sends us %s BTC", tx.getHashAsString(), Utils.bitcoinValueToFriendlyString(valueSentFromMe), Utils.bitcoinValueToFriendlyString(valueSentToMe)));
            }
            if (tx.getConfidence().getSource().equals((Object)TransactionConfidence.Source.UNKNOWN)) {
                log.warn("Wallet received transaction with an unknown source. Consider tagging tx!");
            }
            if ((currentConfidence = tx.getConfidence().getConfidenceType()) == TransactionConfidence.ConfidenceType.UNKNOWN) {
                tx.getConfidence().setConfidenceType(TransactionConfidence.ConfidenceType.NOT_SEEN_IN_CHAIN);
                this.invokeOnTransactionConfidenceChanged(tx);
            }
            this.commitTx(tx);
        }
        finally {
            this.lock.unlock();
        }
    }

    private static AnalysisResult analyzeTransactionAndDependencies(Transaction tx, List<Transaction> dependencies) {
        AnalysisResult result = new AnalysisResult();
        if (tx.getLockTime() > 0L) {
            result.timeLocked = tx;
        }
        if (dependencies != null) {
            for (Transaction dep : dependencies) {
                if (dep.getLockTime() <= 0L) continue;
                result.timeLocked = dep;
            }
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean isPendingTransactionRelevant(Transaction tx) throws ScriptException {
        this.lock.lock();
        try {
            EnumSet<WalletTransaction.Pool> containingPools = this.getContainingPools(tx);
            if (!containingPools.equals(EnumSet.noneOf(WalletTransaction.Pool.class))) {
                log.debug("Received tx we already saw in a block or created ourselves: " + tx.getHashAsString());
                boolean bl = false;
                return bl;
            }
            if (!this.isTransactionRelevant(tx)) {
                log.debug("Received tx that isn't relevant to this wallet, discarding.");
                boolean bl = false;
                return bl;
            }
            if (tx.getLockTime() > 0L && !this.acceptTimeLockedTransactions) {
                log.warn("Received transaction {} with a lock time of {}, but not configured to accept these, discarding", (Object)tx.getHashAsString(), (Object)tx.getLockTime());
                boolean bl = false;
                return bl;
            }
            boolean bl = true;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean isTransactionRelevant(Transaction tx) throws ScriptException {
        this.lock.lock();
        try {
            boolean bl = tx.getValueSentFromMe(this).compareTo(BigInteger.ZERO) > 0 || tx.getValueSentToMe(this).compareTo(BigInteger.ZERO) > 0 || this.checkForDoubleSpendAgainstPending(tx, false);
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    private boolean checkForDoubleSpendAgainstPending(Transaction tx, boolean takeAction) {
        Preconditions.checkState((boolean)this.lock.isLocked());
        HashSet<TransactionOutPoint> outpoints = new HashSet<TransactionOutPoint>();
        for (TransactionInput input : tx.getInputs()) {
            outpoints.add(input.getOutpoint());
        }
        for (Transaction p : this.pending.values()) {
            for (TransactionInput input : p.getInputs()) {
                TransactionOutPoint outpoint = input.getOutpoint();
                if (!outpoints.contains(outpoint)) continue;
                if (takeAction) {
                    TransactionInput overridingInput = null;
                    for (TransactionInput txInput : tx.getInputs()) {
                        if (!txInput.getOutpoint().equals(outpoint)) continue;
                        overridingInput = txInput;
                    }
                    this.killTx(tx, (TransactionInput)Preconditions.checkNotNull(overridingInput), p);
                }
                return true;
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void receiveFromBlock(Transaction tx, StoredBlock block, AbstractBlockChain.NewBlockType blockType) throws VerificationException {
        this.lock.lock();
        try {
            this.receive(tx, block, blockType, false);
        }
        finally {
            this.lock.unlock();
        }
    }

    private void receive(Transaction tx, StoredBlock block, AbstractBlockChain.NewBlockType blockType, boolean reorg) throws VerificationException {
        boolean wasPending;
        Preconditions.checkState((boolean)this.lock.isLocked());
        BigInteger prevBalance = this.getBalance();
        Sha256Hash txHash = tx.getHash();
        boolean bestChain = blockType == AbstractBlockChain.NewBlockType.BEST_CHAIN;
        boolean sideChain = blockType == AbstractBlockChain.NewBlockType.SIDE_CHAIN;
        BigInteger valueSentFromMe = tx.getValueSentFromMe(this);
        BigInteger valueSentToMe = tx.getValueSentToMe(this);
        BigInteger valueDifference = valueSentToMe.subtract(valueSentFromMe);
        if (!reorg) {
            log.info("Received tx {} for {} BTC: {}", new Object[]{sideChain ? "on a side chain" : "", Utils.bitcoinValueToFriendlyString(valueDifference), tx.getHashAsString()});
        }
        ++this.onWalletChangedSuppressions;
        Transaction wtx = this.pending.remove(txHash);
        if (wtx != null) {
            tx = wtx;
            log.info("  <-pending");
            if (bestChain) {
                if (tx.isEveryOwnedOutputSpent(this)) {
                    log.info("  ->spent");
                    this.addWalletTransaction(WalletTransaction.Pool.SPENT, tx);
                } else {
                    log.info("  ->unspent");
                    this.addWalletTransaction(WalletTransaction.Pool.UNSPENT, tx);
                }
            } else if (sideChain) {
                boolean alreadyPresent;
                log.info("  ->inactive");
                boolean bl = alreadyPresent = this.inactive.put(tx.getHash(), tx) != null;
                if (alreadyPresent) {
                    log.info("Saw a transaction be incorporated into multiple independent side chains");
                }
                this.pending.put(tx.getHash(), tx);
            }
        } else if (sideChain) {
            if (!this.unspent.containsKey(tx.getHash()) && !this.spent.containsKey(tx.getHash())) {
                log.info("  ->inactive");
                this.addWalletTransaction(WalletTransaction.Pool.INACTIVE, tx);
            }
        } else if (bestChain) {
            this.processTxFromBestChain(tx);
        }
        if (block != null) {
            tx.setBlockAppearance(block, bestChain);
            if (bestChain) {
                this.ignoreNextNewBlock.add(txHash);
            }
        }
        BigInteger newBalance = this.getBalance();
        log.info("Balance is now: " + Utils.bitcoinValueToFriendlyString(newBalance));
        boolean bl = wasPending = wtx != null;
        if (!reorg && bestChain && !wasPending) {
            int diff = valueDifference.compareTo(BigInteger.ZERO);
            if (diff > 0) {
                this.invokeOnCoinsReceived(tx, prevBalance, newBalance);
            } else if (diff < 0) {
                this.invokeOnCoinsSent(tx, prevBalance, newBalance);
            } else {
                this.invokeOnWalletChanged();
            }
        }
        --this.onWalletChangedSuppressions;
        Preconditions.checkState((boolean)this.isConsistent());
        this.queueAutoSave();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void notifyNewBestBlock(StoredBlock block) throws VerificationException {
        Sha256Hash newBlockHash = block.getHeader().getHash();
        if (newBlockHash.equals(this.getLastBlockSeenHash())) {
            return;
        }
        this.lock.lock();
        try {
            this.setLastBlockSeenHash(newBlockHash);
            this.setLastBlockSeenHeight(block.getHeight());
            ++this.onWalletChangedSuppressions;
            Set<Transaction> transactions = this.getTransactions(true, false);
            for (Transaction tx : transactions) {
                if (this.ignoreNextNewBlock.contains(tx.getHash())) {
                    this.ignoreNextNewBlock.remove(tx.getHash());
                    continue;
                }
                tx.getConfidence().notifyWorkDone(block.getHeader());
            }
            this.queueAutoSave();
            --this.onWalletChangedSuppressions;
            this.invokeOnWalletChanged();
        }
        finally {
            this.lock.unlock();
        }
    }

    private void processTxFromBestChain(Transaction tx) throws VerificationException {
        boolean hasOutputsToMe;
        boolean isDeadCoinbase;
        Preconditions.checkState((boolean)this.lock.isLocked());
        boolean bl = isDeadCoinbase = tx.isCoinBase() && this.dead.containsKey(tx.getHash());
        if (isDeadCoinbase) {
            log.info("  coinbase tx {} <-dead: confidence {}", (Object)tx.getHashAsString(), (Object)tx.getConfidence().getConfidenceType().name());
            this.dead.remove(tx.getHash());
        }
        if (this.inactive.containsKey(tx.getHash())) {
            log.info("  new tx {} <-inactive", (Object)tx.getHashAsString());
            this.inactive.remove(tx.getHash());
        }
        this.updateForSpends(tx, true);
        boolean bl2 = hasOutputsToMe = tx.getValueSentToMe(this, true).compareTo(BigInteger.ZERO) > 0;
        if (hasOutputsToMe) {
            if (tx.isEveryOwnedOutputSpent(this)) {
                log.info("  new tx {} ->spent (by pending)", (Object)tx.getHashAsString());
                this.addWalletTransaction(WalletTransaction.Pool.SPENT, tx);
            } else {
                log.info("  new tx {} ->unspent", (Object)tx.getHashAsString());
                this.addWalletTransaction(WalletTransaction.Pool.UNSPENT, tx);
            }
        } else if (tx.getValueSentFromMe(this).compareTo(BigInteger.ZERO) > 0) {
            log.info("  new tx {} ->spent", (Object)tx.getHashAsString());
            this.addWalletTransaction(WalletTransaction.Pool.SPENT, tx);
        }
        this.checkForDoubleSpendAgainstPending(tx, true);
    }

    private void updateForSpends(Transaction tx, boolean fromChain) throws VerificationException {
        Preconditions.checkState((boolean)this.lock.isLocked());
        for (TransactionInput input : tx.getInputs()) {
            TransactionInput.ConnectionResult result = input.connect(this.unspent, TransactionInput.ConnectMode.ABORT_ON_CONFLICT);
            if (result == TransactionInput.ConnectionResult.NO_SUCH_TX && (result = input.connect(this.spent, TransactionInput.ConnectMode.ABORT_ON_CONFLICT)) == TransactionInput.ConnectionResult.NO_SUCH_TX && (result = input.connect(this.pending, TransactionInput.ConnectMode.ABORT_ON_CONFLICT)) == TransactionInput.ConnectionResult.NO_SUCH_TX) continue;
            if (result == TransactionInput.ConnectionResult.ALREADY_SPENT) {
                if (fromChain) continue;
                log.warn("Saw double spend from another pending transaction, ignoring tx {}", (Object)tx.getHashAsString());
                log.warn("  offending input is input {}", (Object)tx.getInputs().indexOf(input));
                continue;
            }
            if (result != TransactionInput.ConnectionResult.SUCCESS) continue;
            Transaction connected = (Transaction)Preconditions.checkNotNull((Object)input.getOutpoint().fromTx);
            this.maybeMovePool(connected, "prevtx");
        }
        if (fromChain) {
            for (Transaction pendingTx : this.pending.values()) {
                for (TransactionInput input : pendingTx.getInputs()) {
                    TransactionInput.ConnectionResult result = input.connect(tx, TransactionInput.ConnectMode.ABORT_ON_CONFLICT);
                    Preconditions.checkState((result != TransactionInput.ConnectionResult.ALREADY_SPENT ? 1 : 0) != 0);
                    if (result != TransactionInput.ConnectionResult.SUCCESS) continue;
                    log.info("Connected pending tx input {}:{}", (Object)pendingTx.getHashAsString(), (Object)pendingTx.getInputs().indexOf(input));
                }
            }
        }
    }

    private void killTx(Transaction overridingTx, TransactionInput overridingInput, Transaction killedTx) {
        TransactionOutPoint overriddenOutPoint = overridingInput.getOutpoint();
        log.warn("Saw double spend of {} from chain override pending tx {}", (Object)overriddenOutPoint, (Object)killedTx.getHashAsString());
        log.warn("  <-pending ->dead   killed by {}", (Object)overridingTx.getHashAsString());
        this.pending.remove(killedTx.getHash());
        this.addWalletTransaction(WalletTransaction.Pool.DEAD, killedTx);
        log.info("Disconnecting inputs of the newly dead tx");
        for (TransactionInput deadInput : killedTx.getInputs()) {
            Transaction connected = deadInput.getOutpoint().fromTx;
            if (connected == null) continue;
            deadInput.disconnect();
            this.maybeMovePool(connected, "kill");
        }
        log.info("Trying to connect overriding tx back");
        TransactionInput.ConnectionResult result = overridingInput.connect(this.unspent, TransactionInput.ConnectMode.DISCONNECT_ON_CONFLICT);
        if (result == TransactionInput.ConnectionResult.SUCCESS) {
            this.maybeMovePool(overridingInput.getOutpoint().fromTx, "kill");
        } else {
            result = overridingInput.connect(this.spent, TransactionInput.ConnectMode.DISCONNECT_ON_CONFLICT);
            if (result == TransactionInput.ConnectionResult.SUCCESS) {
                this.maybeMovePool(overridingInput.getOutpoint().fromTx, "kill");
            }
        }
        log.info("Informing tx listeners of double spend event");
        killedTx.getConfidence().setOverridingTransaction(overridingTx);
    }

    private void maybeMovePool(Transaction tx, String context) {
        Preconditions.checkState((boolean)this.lock.isLocked());
        if (tx.isEveryOwnedOutputSpent(this)) {
            if (this.unspent.remove(tx.getHash()) != null) {
                if (log.isInfoEnabled()) {
                    log.info("  {} {} <-unspent ->spent", (Object)tx.getHashAsString(), (Object)context);
                }
                this.spent.put(tx.getHash(), tx);
            }
        } else if (this.spent.remove(tx.getHash()) != null) {
            if (log.isInfoEnabled()) {
                log.info("  {} {} <-spent ->unspent", (Object)tx.getHashAsString(), (Object)context);
            }
            this.unspent.put(tx.getHash(), tx);
        }
    }

    public void addEventListener(WalletEventListener listener) {
        this.eventListeners.add(listener);
    }

    public boolean removeEventListener(WalletEventListener listener) {
        return this.eventListeners.remove(listener);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void commitTx(Transaction tx) throws VerificationException {
        this.lock.lock();
        try {
            Preconditions.checkArgument((!this.pending.containsKey(tx.getHash()) ? 1 : 0) != 0, (Object)"commitTx called on the same transaction twice");
            log.info("commitTx of {}", (Object)tx.getHashAsString());
            BigInteger balance = this.getBalance();
            tx.setUpdateTime(Utils.now());
            this.updateForSpends(tx, false);
            log.info("->pending: {}", (Object)tx.getHashAsString());
            this.addWalletTransaction(WalletTransaction.Pool.PENDING, tx);
            try {
                BigInteger valueSentFromMe = tx.getValueSentFromMe(this);
                BigInteger valueSentToMe = tx.getValueSentToMe(this);
                BigInteger newBalance = balance.add(valueSentToMe).subtract(valueSentFromMe);
                if (valueSentToMe.compareTo(BigInteger.ZERO) > 0) {
                    this.invokeOnCoinsReceived(tx, balance, newBalance);
                }
                if (valueSentFromMe.compareTo(BigInteger.ZERO) > 0) {
                    this.invokeOnCoinsSent(tx, balance, newBalance);
                }
                this.invokeOnWalletChanged();
            }
            catch (ScriptException e) {
                throw new RuntimeException(e);
            }
            Preconditions.checkState((boolean)this.isConsistent());
            this.queueAutoSave();
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<Transaction> getTransactions(boolean includeDead, boolean includeInactive) {
        this.lock.lock();
        try {
            HashSet<Transaction> all = new HashSet<Transaction>();
            all.addAll(this.unspent.values());
            all.addAll(this.spent.values());
            all.addAll(this.pending.values());
            if (includeDead) {
                all.addAll(this.dead.values());
            }
            if (includeInactive) {
                all.addAll(this.inactive.values());
            }
            HashSet<Transaction> hashSet = all;
            return hashSet;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Iterable<WalletTransaction> getWalletTransactions() {
        this.lock.lock();
        try {
            HashSet<Transaction> pendingInactive = new HashSet<Transaction>();
            pendingInactive.addAll(this.pending.values());
            pendingInactive.retainAll(this.inactive.values());
            HashSet<Transaction> onlyPending = new HashSet<Transaction>();
            HashSet<Transaction> onlyInactive = new HashSet<Transaction>();
            onlyPending.addAll(this.pending.values());
            onlyPending.removeAll(pendingInactive);
            onlyInactive.addAll(this.inactive.values());
            onlyInactive.removeAll(pendingInactive);
            HashSet<WalletTransaction> all = new HashSet<WalletTransaction>();
            Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.UNSPENT, this.unspent.values());
            Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.SPENT, this.spent.values());
            Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.DEAD, this.dead.values());
            Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.PENDING, onlyPending);
            Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.INACTIVE, onlyInactive);
            Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.PENDING_INACTIVE, pendingInactive);
            HashSet<WalletTransaction> hashSet = all;
            return hashSet;
        }
        finally {
            this.lock.unlock();
        }
    }

    private static void addWalletTransactionsToSet(Set<WalletTransaction> txs, WalletTransaction.Pool poolType, Collection<Transaction> pool) {
        for (Transaction tx : pool) {
            txs.add(new WalletTransaction(poolType, tx));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addWalletTransaction(WalletTransaction wtx) {
        this.lock.lock();
        try {
            this.addWalletTransaction(wtx.getPool(), wtx.getTransaction());
        }
        finally {
            this.lock.unlock();
        }
    }

    private void addWalletTransaction(WalletTransaction.Pool pool, Transaction tx) {
        Preconditions.checkState((boolean)this.lock.isLocked());
        switch (pool) {
            case UNSPENT: {
                Preconditions.checkState((this.unspent.put(tx.getHash(), tx) == null ? 1 : 0) != 0);
                break;
            }
            case SPENT: {
                Preconditions.checkState((this.spent.put(tx.getHash(), tx) == null ? 1 : 0) != 0);
                break;
            }
            case PENDING: {
                Preconditions.checkState((this.pending.put(tx.getHash(), tx) == null ? 1 : 0) != 0);
                break;
            }
            case DEAD: {
                Preconditions.checkState((this.dead.put(tx.getHash(), tx) == null ? 1 : 0) != 0);
                break;
            }
            case INACTIVE: {
                Preconditions.checkState((this.inactive.put(tx.getHash(), tx) == null ? 1 : 0) != 0);
                break;
            }
            case PENDING_INACTIVE: {
                Preconditions.checkState((this.pending.put(tx.getHash(), tx) == null ? 1 : 0) != 0);
                Preconditions.checkState((this.inactive.put(tx.getHash(), tx) == null ? 1 : 0) != 0);
                break;
            }
            default: {
                throw new RuntimeException("Unknown wallet transaction type " + (Object)((Object)pool));
            }
        }
        tx.getConfidence().addEventListener(this.txConfidenceListener);
    }

    public List<Transaction> getTransactionsByTime() {
        return this.getRecentTransactions(0, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<Transaction> getRecentTransactions(int numTransactions, boolean includeDead) {
        this.lock.lock();
        try {
            Preconditions.checkArgument((numTransactions >= 0 ? 1 : 0) != 0);
            int size = this.getPoolSize(WalletTransaction.Pool.UNSPENT) + this.getPoolSize(WalletTransaction.Pool.SPENT) + this.getPoolSize(WalletTransaction.Pool.PENDING);
            if (numTransactions > size || numTransactions == 0) {
                numTransactions = size;
            }
            ArrayList<Transaction> all = new ArrayList<Transaction>(this.getTransactions(includeDead, false));
            Collections.sort(all, Collections.reverseOrder(new Comparator<Transaction>(){

                @Override
                public int compare(Transaction t1, Transaction t2) {
                    return t1.getUpdateTime().compareTo(t2.getUpdateTime());
                }
            }));
            if (numTransactions == all.size()) {
                ArrayList<Transaction> arrayList = all;
                return arrayList;
            }
            all.subList(numTransactions, all.size()).clear();
            ArrayList<Transaction> arrayList = all;
            return arrayList;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Transaction getTransaction(Sha256Hash hash) {
        this.lock.lock();
        try {
            Transaction tx = this.pending.get(hash);
            if (tx != null) {
                Transaction transaction = tx;
                return transaction;
            }
            tx = this.unspent.get(hash);
            if (tx != null) {
                Transaction transaction = tx;
                return transaction;
            }
            tx = this.spent.get(hash);
            if (tx != null) {
                Transaction transaction = tx;
                return transaction;
            }
            tx = this.inactive.get(hash);
            if (tx != null) {
                Transaction transaction = tx;
                return transaction;
            }
            tx = this.dead.get(hash);
            if (tx != null) {
                Transaction transaction = tx;
                return transaction;
            }
            Transaction transaction = null;
            return transaction;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clearTransactions(int fromHeight) {
        block4: {
            this.lock.lock();
            try {
                if (fromHeight == 0) {
                    this.unspent.clear();
                    this.spent.clear();
                    this.pending.clear();
                    this.inactive.clear();
                    this.dead.clear();
                    this.queueAutoSave();
                    break block4;
                }
                throw new UnsupportedOperationException();
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    EnumSet<WalletTransaction.Pool> getContainingPools(Transaction tx) {
        this.lock.lock();
        try {
            EnumSet<WalletTransaction.Pool> result = EnumSet.noneOf(WalletTransaction.Pool.class);
            Sha256Hash txHash = tx.getHash();
            if (this.unspent.containsKey(txHash)) {
                result.add(WalletTransaction.Pool.UNSPENT);
            }
            if (this.spent.containsKey(txHash)) {
                result.add(WalletTransaction.Pool.SPENT);
            }
            if (this.pending.containsKey(txHash)) {
                result.add(WalletTransaction.Pool.PENDING);
            }
            if (this.inactive.containsKey(txHash)) {
                result.add(WalletTransaction.Pool.INACTIVE);
            }
            if (this.dead.containsKey(txHash)) {
                result.add(WalletTransaction.Pool.DEAD);
            }
            EnumSet<WalletTransaction.Pool> enumSet = result;
            return enumSet;
        }
        finally {
            this.lock.unlock();
        }
    }

    int getPoolSize(WalletTransaction.Pool pool) {
        this.lock.lock();
        try {
            switch (pool) {
                case UNSPENT: {
                    int n = this.unspent.size();
                    return n;
                }
                case SPENT: {
                    int n = this.spent.size();
                    return n;
                }
                case PENDING: {
                    int n = this.pending.size();
                    return n;
                }
                case INACTIVE: {
                    int n = this.inactive.size();
                    return n;
                }
                case DEAD: {
                    int n = this.dead.size();
                    return n;
                }
                case ALL: {
                    int n = this.unspent.size() + this.spent.size() + this.pending.size() + this.inactive.size() + this.dead.size();
                    return n;
                }
            }
            throw new RuntimeException("Unreachable");
        }
        finally {
            this.lock.unlock();
        }
    }

    public Transaction createSend(Address address, BigInteger nanocoins) {
        SendRequest req = SendRequest.to(address, nanocoins);
        if (this.completeTx(req)) {
            return req.tx;
        }
        return null;
    }

    public Transaction sendCoinsOffline(SendRequest request) {
        this.lock.lock();
        try {
            if (!this.completeTx(request)) {
                Transaction transaction = null;
                return transaction;
            }
            this.commitTx(request.tx);
            Transaction transaction = request.tx;
            return transaction;
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
        finally {
            this.lock.unlock();
        }
    }

    public SendResult sendCoins(PeerGroup peerGroup, Address to, BigInteger value) {
        SendRequest request = SendRequest.to(to, value);
        return this.sendCoins(peerGroup, request);
    }

    public SendResult sendCoins(PeerGroup peerGroup, SendRequest request) {
        Transaction tx = this.sendCoinsOffline(request);
        if (tx == null) {
            return null;
        }
        SendResult result = new SendResult();
        result.tx = tx;
        result.broadcastComplete = peerGroup.broadcastTransaction(tx);
        return result;
    }

    public Transaction sendCoins(Peer peer, SendRequest request) throws IOException {
        Transaction tx = this.sendCoinsOffline(request);
        if (tx == null) {
            return null;
        }
        peer.sendMessage(tx);
        return tx;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean completeTx(SendRequest req) {
        this.lock.lock();
        try {
            Preconditions.checkArgument((!req.completed ? 1 : 0) != 0, (Object)"Given SendRequest has already been completed.");
            BigInteger value = BigInteger.ZERO;
            for (TransactionOutput output : req.tx.getOutputs()) {
                value = value.add(output.getValue());
            }
            value = value.add(req.fee);
            log.info("Completing send tx with {} outputs totalling {}", (Object)req.tx.getOutputs().size(), (Object)Utils.bitcoinValueToFriendlyString(value));
            LinkedList<TransactionOutput> candidates = this.calculateSpendCandidates(true);
            CoinSelection selection = this.coinSelector.select(value, candidates);
            if (selection.valueGathered.compareTo(value) < 0) {
                log.warn("Insufficient value in wallet for send, missing " + Utils.bitcoinValueToFriendlyString(value.subtract(selection.valueGathered)));
                boolean bl = false;
                return bl;
            }
            Preconditions.checkState((selection.gathered.size() > 0 ? 1 : 0) != 0);
            req.tx.getConfidence().setConfidenceType(TransactionConfidence.ConfidenceType.NOT_SEEN_IN_CHAIN);
            BigInteger change = selection.valueGathered.subtract(value);
            if (change.compareTo(BigInteger.ZERO) > 0) {
                Address changeAddress = req.changeAddress != null ? req.changeAddress : this.getChangeAddress();
                log.info("  with {} coins change", (Object)Utils.bitcoinValueToFriendlyString(change));
                req.tx.addOutput(new TransactionOutput(this.params, req.tx, change, changeAddress));
            }
            for (TransactionOutput output : selection.gathered) {
                req.tx.addInput(output);
            }
            try {
                req.tx.signInputs(Transaction.SigHash.ALL, this, req.aesKey);
            }
            catch (ScriptException e) {
                throw new RuntimeException(e);
            }
            int size = req.tx.bitcoinSerialize().length;
            if (size > 102400) {
                log.error("Transaction could not be created without exceeding max size: {} vs {}", (Object)size, (Object)102400);
                boolean bl = false;
                return bl;
            }
            req.tx.getConfidence().setSource(TransactionConfidence.Source.SELF);
            req.completed = true;
            log.info("  completed {} with {} inputs", (Object)req.tx.getHashAsString(), (Object)req.tx.getInputs().size());
            boolean bl = true;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    private LinkedList<TransactionOutput> calculateSpendCandidates(boolean excludeImmatureCoinbases) {
        Preconditions.checkState((boolean)this.lock.isLocked());
        LinkedList candidates = Lists.newLinkedList();
        for (Transaction tx : Iterables.concat(this.unspent.values(), this.pending.values())) {
            if (excludeImmatureCoinbases && !tx.isMature()) continue;
            for (TransactionOutput output : tx.getOutputs()) {
                if (!output.isAvailableForSpending() || !output.isMine(this)) continue;
                candidates.add(output);
            }
        }
        return candidates;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Address getChangeAddress() {
        this.lock.lock();
        try {
            Preconditions.checkState((this.keychain.size() > 0 ? 1 : 0) != 0, (Object)"Can't send value without an address to use for receiving change");
            ECKey first = this.keychain.get(0);
            Address address = first.toAddress(this.params);
            return address;
        }
        finally {
            this.lock.unlock();
        }
    }

    public boolean addKey(ECKey key) {
        return this.addKeys(Lists.newArrayList((Object[])new ECKey[]{key})) == 1;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int addKeys(List<ECKey> keys) {
        int added = 0;
        this.lock.lock();
        try {
            for (ECKey key : keys) {
                if (this.keychain.contains(key)) continue;
                if (this.keyCrypter != null && this.keyCrypter.getUnderstoodEncryptionType() != Protos.Wallet.EncryptionType.UNENCRYPTED && key.isEncrypted() && !this.keyCrypter.equals(key.getKeyCrypter())) {
                    throw new KeyCrypterException("Cannot add key " + key.toString() + " because the keyCrypter does not match the wallets. Keys must be homogenous.");
                }
                this.keychain.add(key);
                ++added;
            }
            if (this.autosaveToFile != null) {
                this.autoSave();
            }
        }
        finally {
            this.lock.unlock();
        }
        for (ECKey key : keys) {
            for (WalletEventListener listener : this.eventListeners) {
                listener.onKeyAdded(key);
            }
        }
        return added;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ECKey findKeyFromPubHash(byte[] pubkeyHash) {
        this.lock.lock();
        try {
            for (ECKey key : this.keychain) {
                if (!Arrays.equals(key.getPubKeyHash(), pubkeyHash)) continue;
                ECKey eCKey = key;
                return eCKey;
            }
            ECKey eCKey = null;
            return eCKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean hasKey(ECKey key) {
        this.lock.lock();
        try {
            boolean bl = this.keychain.contains(key);
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    public boolean isPubKeyHashMine(byte[] pubkeyHash) {
        return this.findKeyFromPubHash(pubkeyHash) != null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ECKey findKeyFromPubKey(byte[] pubkey) {
        this.lock.lock();
        try {
            for (ECKey key : this.keychain) {
                if (!Arrays.equals(key.getPubKey(), pubkey)) continue;
                ECKey eCKey = key;
                return eCKey;
            }
            ECKey eCKey = null;
            return eCKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    public boolean isPubKeyMine(byte[] pubkey) {
        return this.findKeyFromPubKey(pubkey) != null;
    }

    public BigInteger getBalance() {
        return this.getBalance(BalanceType.AVAILABLE);
    }

    public BigInteger getBalance(BalanceType balanceType) {
        this.lock.lock();
        try {
            if (balanceType == BalanceType.AVAILABLE) {
                BigInteger bigInteger = this.getBalance(this.coinSelector);
                return bigInteger;
            }
            if (balanceType == BalanceType.ESTIMATED) {
                LinkedList<TransactionOutput> all = this.calculateSpendCandidates(false);
                BigInteger value = BigInteger.ZERO;
                for (TransactionOutput out : all) {
                    value = value.add(out.getValue());
                }
                BigInteger bigInteger = value;
                return bigInteger;
            }
            throw new AssertionError((Object)"Unknown balance type");
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public BigInteger getBalance(CoinSelector selector) {
        this.lock.lock();
        try {
            Preconditions.checkNotNull((Object)selector);
            LinkedList<TransactionOutput> candidates = this.calculateSpendCandidates(true);
            CoinSelection selection = selector.select(NetworkParameters.MAX_MONEY, candidates);
            BigInteger bigInteger = selection.valueGathered;
            return bigInteger;
        }
        finally {
            this.lock.unlock();
        }
    }

    public String toString() {
        return this.toString(false, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String toString(boolean includePrivateKeys, AbstractBlockChain chain) {
        this.lock.lock();
        try {
            StringBuilder builder = new StringBuilder();
            builder.append(String.format("Wallet containing %s BTC in:%n", Utils.bitcoinValueToFriendlyString(this.getBalance())));
            builder.append(String.format("  %d unspent transactions%n", this.unspent.size()));
            builder.append(String.format("  %d spent transactions%n", this.spent.size()));
            builder.append(String.format("  %d pending transactions%n", this.pending.size()));
            builder.append(String.format("  %d inactive transactions%n", this.inactive.size()));
            builder.append(String.format("  %d dead transactions%n", this.dead.size()));
            builder.append(String.format("Last seen best block: (%d) %s%n", this.getLastBlockSeenHeight(), this.getLastBlockSeenHash()));
            if (this.keyCrypter != null) {
                builder.append(String.format("Encryption: %s%n", this.keyCrypter.toString()));
            }
            builder.append("\nKeys:\n");
            for (ECKey key : this.keychain) {
                builder.append("  addr:");
                builder.append(key.toAddress(this.params));
                builder.append(" ");
                builder.append(includePrivateKeys ? key.toStringWithPrivate() : key.toString());
                builder.append("\n");
            }
            if (this.unspent.size() > 0) {
                builder.append("\nUNSPENT:\n");
                this.toStringHelper(builder, this.unspent, chain);
            }
            if (this.spent.size() > 0) {
                builder.append("\nSPENT:\n");
                this.toStringHelper(builder, this.spent, chain);
            }
            if (this.pending.size() > 0) {
                builder.append("\nPENDING:\n");
                this.toStringHelper(builder, this.pending, chain);
            }
            if (this.inactive.size() > 0) {
                builder.append("\nINACTIVE:\n");
                this.toStringHelper(builder, this.inactive, chain);
            }
            if (this.dead.size() > 0) {
                builder.append("\nDEAD:\n");
                this.toStringHelper(builder, this.dead, chain);
            }
            String string = builder.toString();
            return string;
        }
        finally {
            this.lock.unlock();
        }
    }

    private void toStringHelper(StringBuilder builder, Map<Sha256Hash, Transaction> transactionMap, AbstractBlockChain chain) {
        Preconditions.checkState((boolean)this.lock.isLocked());
        for (Transaction tx : transactionMap.values()) {
            try {
                builder.append("Sends ");
                builder.append(Utils.bitcoinValueToFriendlyString(tx.getValueSentFromMe(this)));
                builder.append(" and receives ");
                builder.append(Utils.bitcoinValueToFriendlyString(tx.getValueSentToMe(this)));
                builder.append(", total value ");
                builder.append(Utils.bitcoinValueToFriendlyString(tx.getValue(this)));
                builder.append(".\n");
            }
            catch (ScriptException e) {
                // empty catch block
            }
            builder.append(tx.toString(chain));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void reorganize(StoredBlock splitPoint, List<StoredBlock> oldBlocks, List<StoredBlock> newBlocks) throws VerificationException {
        this.lock.lock();
        try {
            ArrayList<Sha256Hash> oldBlockHashes = new ArrayList<Sha256Hash>(oldBlocks.size());
            ArrayList<Sha256Hash> newBlockHashes = new ArrayList<Sha256Hash>(newBlocks.size());
            log.info("Old part of chain (top to bottom):");
            for (StoredBlock b : oldBlocks) {
                log.info("  {}", (Object)b.getHeader().getHashAsString());
                oldBlockHashes.add(b.getHeader().getHash());
            }
            log.info("New part of chain (top to bottom):");
            for (StoredBlock b : newBlocks) {
                log.info("  {}", (Object)b.getHeader().getHashAsString());
                newBlockHashes.add(b.getHeader().getHash());
            }
            HashMap<Sha256Hash, Transaction> oldChainTransactions = new HashMap<Sha256Hash, Transaction>();
            HashMap<Sha256Hash, Transaction> onlyOldChainTransactions = new HashMap<Sha256Hash, Transaction>();
            HashMap<Sha256Hash, Transaction> newChainTransactions = new HashMap<Sha256Hash, Transaction>();
            HashMap<Sha256Hash, Transaction> commonChainTransactions = new HashMap<Sha256Hash, Transaction>();
            HashMap<Sha256Hash, Transaction> all = new HashMap<Sha256Hash, Transaction>();
            all.putAll(this.unspent);
            all.putAll(this.spent);
            all.putAll(this.inactive);
            for (Transaction tx : this.dead.values()) {
                if (!tx.isCoinBase()) continue;
                all.put(tx.getHash(), tx);
            }
            for (Transaction tx : all.values()) {
                boolean alreadyPresent;
                boolean inCommonSection;
                Collection<Sha256Hash> appearsIn = tx.getAppearsInHashes();
                Preconditions.checkNotNull(appearsIn);
                boolean inOldSection = !Collections.disjoint(appearsIn, oldBlockHashes);
                boolean inNewSection = !Collections.disjoint(appearsIn, newBlockHashes);
                boolean bl = inCommonSection = !inNewSection && !inOldSection;
                if (inCommonSection) {
                    alreadyPresent = commonChainTransactions.put(tx.getHash(), tx) != null;
                    Preconditions.checkState((!alreadyPresent ? 1 : 0) != 0, (Object)"Transaction appears twice in common chain segment");
                    continue;
                }
                if (inOldSection) {
                    alreadyPresent = oldChainTransactions.put(tx.getHash(), tx) != null;
                    Preconditions.checkState((!alreadyPresent ? 1 : 0) != 0, (Object)"Transaction appears twice in old chain segment");
                    if (!inNewSection) {
                        alreadyPresent = onlyOldChainTransactions.put(tx.getHash(), tx) != null;
                        Preconditions.checkState((!alreadyPresent ? 1 : 0) != 0, (Object)"Transaction appears twice in only-old map");
                    }
                }
                if (!inNewSection) continue;
                alreadyPresent = newChainTransactions.put(tx.getHash(), tx) != null;
                Preconditions.checkState((!alreadyPresent ? 1 : 0) != 0, (Object)"Transaction appears twice in new chain segment");
            }
            boolean affectedUs = !oldChainTransactions.equals(newChainTransactions);
            log.info(affectedUs ? "Re-org affected our transactions" : "Re-org had no effect on our transactions");
            if (!affectedUs) {
                return;
            }
            ++this.onWalletChangedSuppressions;
            for (Transaction tx : onlyOldChainTransactions.values()) {
                log.info("  Only Old: {}", (Object)tx.getHashAsString());
            }
            for (Transaction tx : oldChainTransactions.values()) {
                log.info("  Old: {}", (Object)tx.getHashAsString());
            }
            for (Transaction tx : newChainTransactions.values()) {
                log.info("  New: {}", (Object)tx.getHashAsString());
            }
            for (Transaction tx : all.values()) {
                tx.disconnectInputs();
            }
            for (Transaction tx : this.pending.values()) {
                tx.disconnectInputs();
            }
            for (Transaction tx : commonChainTransactions.values()) {
                TransactionInput badInput = tx.connectForReorganize(all);
                Preconditions.checkState((badInput == null ? 1 : 0) != 0, (String)"Failed to connect %s, %s", (Object[])new Object[]{tx.getHashAsString(), badInput == null ? "" : badInput.toString()});
            }
            log.info("Moving transactions");
            this.unspent.clear();
            this.spent.clear();
            this.inactive.clear();
            for (Transaction tx : commonChainTransactions.values()) {
                int unspentOutputs = 0;
                for (TransactionOutput output : tx.getOutputs()) {
                    if (!output.isAvailableForSpending() || !output.isMine(this)) continue;
                    ++unspentOutputs;
                }
                if (unspentOutputs > 0) {
                    log.info("  TX {} ->unspent", (Object)tx.getHashAsString());
                    this.unspent.put(tx.getHash(), tx);
                    continue;
                }
                log.info("  TX {} ->spent", (Object)tx.getHashAsString());
                this.spent.put(tx.getHash(), tx);
            }
            for (Transaction tx : onlyOldChainTransactions.values()) {
                tx.notifyNotOnBestChain();
                if (!tx.isCoinBase()) continue;
                if (this.unspent.containsKey(tx.getHash())) {
                    log.info("  coinbase tx {} unspent->dead", (Object)tx.getHashAsString());
                    this.unspent.remove(tx.getHash());
                } else if (this.spent.containsKey(tx.getHash())) {
                    log.info("  coinbase tx {} spent->dead", (Object)tx.getHashAsString());
                    this.spent.remove(tx.getHash());
                }
                this.dead.put(tx.getHash(), tx);
                tx.getConfidence().setConfidenceType(TransactionConfidence.ConfidenceType.DEAD);
            }
            Collections.reverse(newBlocks);
            int depthToSubtract = oldBlocks.size();
            BigInteger workDoneToSubtract = BigInteger.ZERO;
            for (StoredBlock b : oldBlocks) {
                workDoneToSubtract = workDoneToSubtract.add(b.getHeader().getWork());
            }
            log.info("DepthToSubtract = " + depthToSubtract + ", workDoneToSubtract = " + workDoneToSubtract);
            Wallet.subtractDepthAndWorkDone(depthToSubtract, workDoneToSubtract, this.spent.values());
            Wallet.subtractDepthAndWorkDone(depthToSubtract, workDoneToSubtract, this.unspent.values());
            Wallet.subtractDepthAndWorkDone(depthToSubtract, workDoneToSubtract, this.dead.values());
            this.setLastBlockSeenHash(splitPoint.getHeader().getHash());
            for (StoredBlock b : newBlocks) {
                log.info("Replaying block {}", (Object)b.getHeader().getHashAsString());
                HashSet<Transaction> txns = new HashSet<Transaction>();
                Sha256Hash blockHash = b.getHeader().getHash();
                for (Transaction tx : newChainTransactions.values()) {
                    if (!tx.getAppearsInHashes().contains(blockHash)) continue;
                    txns.add(tx);
                    log.info("  containing tx {}", (Object)tx.getHashAsString());
                }
                if (!txns.isEmpty()) {
                    for (Transaction t : txns) {
                        try {
                            this.receive(t, b, AbstractBlockChain.NewBlockType.BEST_CHAIN, true);
                        }
                        catch (ScriptException e) {
                            throw new RuntimeException(e);
                        }
                    }
                }
                this.notifyNewBestBlock(b);
            }
            HashMap<Sha256Hash, Transaction> pool = new HashMap<Sha256Hash, Transaction>();
            pool.putAll(this.unspent);
            pool.putAll(this.spent);
            pool.putAll(this.pending);
            HashMap<Sha256Hash, Transaction> toReprocess = new HashMap<Sha256Hash, Transaction>();
            toReprocess.putAll(onlyOldChainTransactions);
            toReprocess.putAll(this.pending);
            log.info("Reprocessing transactions not in new best chain:");
            for (Transaction tx : this.dead.values()) {
                this.reprocessUnincludedTxAfterReorg(pool, tx);
            }
            for (Transaction tx : toReprocess.values()) {
                this.reprocessUnincludedTxAfterReorg(pool, tx);
            }
            log.info("post-reorg balance is {}", (Object)Utils.bitcoinValueToFriendlyString(this.getBalance()));
            this.invokeOnReorganize();
            --this.onWalletChangedSuppressions;
            this.invokeOnWalletChanged();
            Preconditions.checkState((boolean)this.isConsistent());
        }
        finally {
            this.lock.unlock();
        }
    }

    private static void subtractDepthAndWorkDone(int depthToSubtract, BigInteger workDoneToSubtract, Collection<Transaction> transactions) {
        for (Transaction tx : transactions) {
            if (tx.getConfidence().getConfidenceType() != TransactionConfidence.ConfidenceType.BUILDING) continue;
            tx.getConfidence().setDepthInBlocks(tx.getConfidence().getDepthInBlocks() - depthToSubtract);
            tx.getConfidence().setWorkDone(tx.getConfidence().getWorkDone().subtract(workDoneToSubtract));
        }
    }

    private void reprocessUnincludedTxAfterReorg(Map<Sha256Hash, Transaction> pool, Transaction tx) {
        boolean isDeadCoinbase;
        Preconditions.checkState((boolean)this.lock.isLocked());
        log.info("TX {}", (Object)(tx.getHashAsString() + ", confidence = " + tx.getConfidence().getConfidenceType().name()));
        boolean bl = isDeadCoinbase = tx.isCoinBase() && TransactionConfidence.ConfidenceType.DEAD == tx.getConfidence().getConfidenceType();
        if (isDeadCoinbase) {
            return;
        }
        int numInputs = tx.getInputs().size();
        int noSuchTx = 0;
        int success = 0;
        boolean isDead = false;
        HashSet<Object> connectedTransactions = new HashSet<Object>();
        for (TransactionInput transactionInput : tx.getInputs()) {
            TransactionInput.ConnectionResult result = transactionInput.connect(pool, TransactionInput.ConnectMode.ABORT_ON_CONFLICT);
            if (result == TransactionInput.ConnectionResult.SUCCESS) {
                ++success;
                TransactionOutput connectedOutput = (TransactionOutput)Preconditions.checkNotNull((Object)transactionInput.getConnectedOutput(pool));
                connectedTransactions.add(Preconditions.checkNotNull((Object)connectedOutput.parentTransaction));
                continue;
            }
            if (result == TransactionInput.ConnectionResult.NO_SUCH_TX) {
                ++noSuchTx;
                continue;
            }
            if (result != TransactionInput.ConnectionResult.ALREADY_SPENT) continue;
            isDead = true;
            log.info("   ->dead, will not confirm now unless there's another re-org", (Object)tx.getHashAsString());
            TransactionOutput doubleSpent = transactionInput.getConnectedOutput(pool);
            Transaction replacement = doubleSpent.getSpentBy().getParentTransaction();
            this.dead.put(tx.getHash(), tx);
            this.pending.remove(tx.getHash());
            tx.getConfidence().setOverridingTransaction(replacement);
            break;
        }
        if (isDead) {
            return;
        }
        if (noSuchTx == numInputs) {
            log.info("   ->inactive", (Object)(tx.getHashAsString() + ", confidence = " + tx.getConfidence().getConfidenceType().name()));
            this.inactive.put(tx.getHash(), tx);
            this.dead.remove(tx.getHash());
        } else if (success == numInputs - noSuchTx) {
            log.info("   ->pending", (Object)(tx.getHashAsString() + ", confidence = " + tx.getConfidence().getConfidenceType().name()));
            this.pending.put(tx.getHash(), tx);
            this.dead.remove(tx.getHash());
        }
        for (Transaction transaction : connectedTransactions) {
            this.maybeMovePool(transaction, "reorg");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Collection<Transaction> getPendingTransactions() {
        this.lock.lock();
        try {
            Collection<Transaction> collection = Collections.unmodifiableCollection(this.pending.values());
            return collection;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long getEarliestKeyCreationTime() {
        this.lock.lock();
        try {
            if (this.keychain.size() == 0) {
                long l = Utils.now().getTime() / 1000L;
                return l;
            }
            long earliestTime = Long.MAX_VALUE;
            for (ECKey key : this.keychain) {
                earliestTime = Math.min(key.getCreationTimeSeconds(), earliestTime);
            }
            long l = earliestTime;
            return l;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Sha256Hash getLastBlockSeenHash() {
        this.lock.lock();
        try {
            Sha256Hash sha256Hash = this.lastBlockSeenHash;
            return sha256Hash;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setLastBlockSeenHash(Sha256Hash lastBlockSeenHash) {
        this.lock.lock();
        try {
            this.lastBlockSeenHash = lastBlockSeenHash;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setLastBlockSeenHeight(int lastBlockSeenHeight) {
        this.lock.lock();
        try {
            this.lastBlockSeenHeight = lastBlockSeenHeight;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getLastBlockSeenHeight() {
        this.lock.lock();
        try {
            int n = this.lastBlockSeenHeight;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public KeyParameter encrypt(CharSequence password) {
        Preconditions.checkNotNull((Object)password);
        Preconditions.checkArgument((password.length() > 0 ? 1 : 0) != 0);
        KeyCrypterScrypt scrypt = new KeyCrypterScrypt();
        KeyParameter derivedKey = scrypt.deriveKey(password);
        this.encrypt(scrypt, derivedKey);
        return derivedKey;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void encrypt(KeyCrypter keyCrypter, KeyParameter aesKey) {
        this.lock.lock();
        try {
            Preconditions.checkNotNull((Object)keyCrypter);
            Preconditions.checkState((this.getEncryptionType() == Protos.Wallet.EncryptionType.UNENCRYPTED ? 1 : 0) != 0, (Object)"Wallet is already encrypted");
            ArrayList<ECKey> encryptedKeyChain = new ArrayList<ECKey>();
            for (ECKey key : this.keychain) {
                if (key.isEncrypted()) {
                    encryptedKeyChain.add(key);
                    continue;
                }
                ECKey encryptedKey = key.encrypt(keyCrypter, aesKey);
                if (!ECKey.encryptionIsReversible(key, encryptedKey, keyCrypter, aesKey)) {
                    throw new KeyCrypterException("The key " + key.toString() + " cannot be successfully decrypted after encryption so aborting wallet encryption.");
                }
                encryptedKeyChain.add(encryptedKey);
            }
            for (ECKey key : this.keychain) {
                if (key.isEncrypted()) continue;
                key.clearPrivateKey();
            }
            this.keychain = encryptedKeyChain;
            this.keyCrypter = keyCrypter;
            if (this.autosaveToFile != null) {
                this.autoSave();
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void decrypt(KeyParameter aesKey) {
        this.lock.lock();
        try {
            Preconditions.checkState((this.getEncryptionType() != Protos.Wallet.EncryptionType.UNENCRYPTED ? 1 : 0) != 0, (Object)"Wallet is already decrypted");
            Preconditions.checkNotNull((Object)this.keyCrypter);
            ArrayList<ECKey> decryptedKeyChain = new ArrayList<ECKey>();
            for (ECKey key : this.keychain) {
                if (!key.isEncrypted()) {
                    decryptedKeyChain.add(key);
                    continue;
                }
                ECKey decryptedECKey = key.decrypt(this.keyCrypter, aesKey);
                decryptedKeyChain.add(decryptedECKey);
            }
            this.keychain = decryptedKeyChain;
            this.keyCrypter = null;
            if (this.autosaveToFile != null) {
                this.autoSave();
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public ECKey addNewEncryptedKey(KeyCrypter keyCrypter, KeyParameter aesKey) {
        ECKey newKey = new ECKey().encrypt((KeyCrypter)Preconditions.checkNotNull((Object)keyCrypter), (KeyParameter)Preconditions.checkNotNull((Object)aesKey));
        this.addKey(newKey);
        return newKey;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ECKey addNewEncryptedKey(CharSequence password) {
        this.lock.lock();
        try {
            Preconditions.checkNotNull((Object)this.keyCrypter, (Object)"Wallet is not encrypted, you must call encrypt() first.");
            ECKey eCKey = this.addNewEncryptedKey(this.keyCrypter, this.keyCrypter.deriveKey(password));
            return eCKey;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean checkPassword(CharSequence password) {
        this.lock.lock();
        try {
            if (this.keyCrypter == null) {
                boolean bl = false;
                return bl;
            }
            boolean bl = this.checkAESKey(this.keyCrypter.deriveKey((CharSequence)Preconditions.checkNotNull((Object)password)));
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Loose catch block
     */
    public boolean checkAESKey(KeyParameter aesKey) {
        this.lock.lock();
        try {
            if (!this.getKeys().iterator().hasNext()) {
                boolean bl = false;
                return bl;
            }
            ECKey firstEncryptedECKey = null;
            Iterator<ECKey> iterator = this.getKeys().iterator();
            while (iterator.hasNext() && firstEncryptedECKey == null) {
                ECKey loopECKey = iterator.next();
                if (!loopECKey.isEncrypted()) continue;
                firstEncryptedECKey = loopECKey;
            }
            if (firstEncryptedECKey == null) {
                boolean loopECKey = false;
                return loopECKey;
            }
            String originalAddress = firstEncryptedECKey.toAddress(this.getNetworkParameters()).toString();
            if (firstEncryptedECKey.isEncrypted() && firstEncryptedECKey.getEncryptedPrivateKey() != null) {
                try {
                    ECKey rebornKey = firstEncryptedECKey.decrypt(this.keyCrypter, aesKey);
                    String rebornAddress = rebornKey.toAddress(this.getNetworkParameters()).toString();
                    boolean bl = originalAddress.equals(rebornAddress);
                    return bl;
                }
                catch (KeyCrypterException ede) {
                    boolean bl = false;
                    this.lock.unlock();
                    return bl;
                }
            }
            boolean bl = false;
            return bl;
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public KeyCrypter getKeyCrypter() {
        this.lock.lock();
        try {
            KeyCrypter keyCrypter = this.keyCrypter;
            return keyCrypter;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Protos.Wallet.EncryptionType getEncryptionType() {
        this.lock.lock();
        try {
            if (this.keyCrypter == null) {
                Protos.Wallet.EncryptionType encryptionType = Protos.Wallet.EncryptionType.UNENCRYPTED;
                return encryptionType;
            }
            Protos.Wallet.EncryptionType encryptionType = this.keyCrypter.getUnderstoodEncryptionType();
            return encryptionType;
        }
        finally {
            this.lock.unlock();
        }
    }

    public boolean isEncrypted() {
        return this.getEncryptionType() != Protos.Wallet.EncryptionType.UNENCRYPTED;
    }

    public int getVersion() {
        return this.version;
    }

    public void setVersion(int version) {
        this.version = version;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public String getDescription() {
        return this.description;
    }

    public int getBloomFilterElementCount() {
        int size = this.getKeychainSize() * 2;
        for (Transaction tx : this.getTransactions(false, true)) {
            for (TransactionOutput out : tx.getOutputs()) {
                try {
                    if (!out.isMine(this) || !out.getScriptPubKey().isSentToRawPubKey()) continue;
                    ++size;
                }
                catch (ScriptException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        return size;
    }

    public BloomFilter getBloomFilter(double falsePositiveRate) {
        return this.getBloomFilter(this.getBloomFilterElementCount(), falsePositiveRate, (long)(Math.random() * 9.223372036854776E18));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public BloomFilter getBloomFilter(int size, double falsePositiveRate, long nTweak) {
        BloomFilter filter = new BloomFilter(size, falsePositiveRate, nTweak);
        this.lock.lock();
        try {
            for (ECKey key : this.keychain) {
                filter.insert(key.getPubKey());
                filter.insert(key.getPubKeyHash());
            }
        }
        finally {
            this.lock.unlock();
        }
        for (Transaction tx : this.getTransactions(false, true)) {
            for (int i = 0; i < tx.getOutputs().size(); ++i) {
                TransactionOutput out = tx.getOutputs().get(i);
                try {
                    if (!out.isMine(this) || !out.getScriptPubKey().isSentToRawPubKey()) continue;
                    TransactionOutPoint outPoint = new TransactionOutPoint(this.params, (long)i, tx);
                    filter.insert(outPoint.bitcoinSerialize());
                    continue;
                }
                catch (ScriptException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        return filter;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CoinSelector getCoinSelector() {
        this.lock.lock();
        try {
            CoinSelector coinSelector = this.coinSelector;
            return coinSelector;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setCoinSelector(CoinSelector coinSelector) {
        this.lock.lock();
        try {
            this.coinSelector = coinSelector;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void invokeOnTransactionConfidenceChanged(Transaction tx) {
        Preconditions.checkState((boolean)this.lock.isLocked());
        this.lock.unlock();
        try {
            for (WalletEventListener listener : this.eventListeners) {
                listener.onTransactionConfidenceChanged(this, tx);
            }
        }
        finally {
            this.lock.lock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void invokeOnWalletChanged() {
        Preconditions.checkState((boolean)this.lock.isLocked());
        Preconditions.checkState((this.onWalletChangedSuppressions >= 0 ? 1 : 0) != 0);
        if (this.onWalletChangedSuppressions > 0) {
            return;
        }
        this.lock.unlock();
        try {
            for (WalletEventListener listener : this.eventListeners) {
                listener.onWalletChanged(this);
            }
        }
        finally {
            this.lock.lock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void invokeOnCoinsReceived(Transaction tx, BigInteger balance, BigInteger newBalance) {
        Preconditions.checkState((boolean)this.lock.isLocked());
        this.lock.unlock();
        try {
            for (WalletEventListener listener : this.eventListeners) {
                listener.onCoinsReceived(this, tx, balance, newBalance);
            }
        }
        finally {
            this.lock.lock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void invokeOnCoinsSent(Transaction tx, BigInteger prevBalance, BigInteger newBalance) {
        Preconditions.checkState((boolean)this.lock.isLocked());
        this.lock.unlock();
        try {
            for (WalletEventListener listener : this.eventListeners) {
                listener.onCoinsSent(this, tx, prevBalance, newBalance);
            }
        }
        finally {
            this.lock.lock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void invokeOnReorganize() {
        Preconditions.checkState((boolean)this.lock.isLocked());
        this.lock.unlock();
        try {
            for (WalletEventListener listener : this.eventListeners) {
                listener.onReorganize(this);
            }
        }
        finally {
            this.lock.lock();
        }
    }

    public static enum BalanceType {
        ESTIMATED,
        AVAILABLE;

    }

    public static class SendRequest {
        public Transaction tx;
        public Address changeAddress = null;
        public BigInteger fee = BigInteger.ZERO;
        public KeyParameter aesKey = null;
        private boolean completed;

        private SendRequest() {
        }

        public static SendRequest to(Address destination, BigInteger value) {
            SendRequest req = new SendRequest();
            req.tx = new Transaction(destination.getParameters());
            req.tx.addOutput(value, destination);
            return req;
        }

        public static SendRequest to(NetworkParameters params, ECKey destination, BigInteger value) {
            SendRequest req = new SendRequest();
            req.tx = new Transaction(params);
            req.tx.addOutput(value, destination);
            return req;
        }

        public static SendRequest forTx(Transaction tx) {
            SendRequest req = new SendRequest();
            req.tx = tx;
            return req;
        }
    }

    public static class SendResult {
        public Transaction tx;
        public ListenableFuture<Transaction> broadcastComplete;
    }

    protected static class AnalysisResult {
        Transaction timeLocked;

        protected AnalysisResult() {
        }
    }

    public static interface AutosaveEventListener {
        public boolean caughtException(Throwable var1);

        public void onBeforeAutoSave(File var1);

        public void onAfterAutoSave(File var1);
    }

    private static class AutosaveThread
    extends Thread {
        private static DelayQueue<WalletSaveRequest> walletRefs = new DelayQueue();
        private static AutosaveThread globalThread;

        private AutosaveThread() {
            this.setDaemon(true);
            this.setName("Wallet auto save thread");
            this.setPriority(1);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public static void maybeStart() {
            if (walletRefs.size() == 0) {
                return;
            }
            Class<AutosaveThread> clazz = AutosaveThread.class;
            synchronized (AutosaveThread.class) {
                if (globalThread == null) {
                    globalThread = new AutosaveThread();
                    globalThread.start();
                }
                // ** MonitorExit[var0] (shouldn't be in output)
                return;
            }
        }

        public static void registerForSave(Wallet wallet, long delayMsec) {
            walletRefs.add(new WalletSaveRequest(wallet, delayMsec));
            AutosaveThread.maybeStart();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            log.info("Auto-save thread starting up");
            try {
                while (true) {
                    WalletSaveRequest req;
                    if ((req = (WalletSaveRequest)walletRefs.poll(5L, TimeUnit.SECONDS)) == null) {
                        if (walletRefs.size() != 0) continue;
                        break;
                    }
                    req.wallet.lock.lock();
                    try {
                        if (!req.wallet.dirty || !req.wallet.autoSave()) continue;
                    }
                    finally {
                        req.wallet.lock.unlock();
                        continue;
                    }
                    break;
                }
            }
            catch (InterruptedException e) {
                log.error("Auto-save thread interrupted during wait", (Throwable)e);
            }
            log.info("Auto-save thread shutting down");
            Class<AutosaveThread> clazz = AutosaveThread.class;
            synchronized (AutosaveThread.class) {
                Preconditions.checkState((globalThread == this ? 1 : 0) != 0);
                globalThread = null;
                // ** MonitorExit[var1_1] (shouldn't be in output)
                AutosaveThread.maybeStart();
                return;
            }
        }

        private static class WalletSaveRequest
        implements Delayed {
            public final Wallet wallet;
            public final long startTimeMs = System.currentTimeMillis();
            public final long requestedDelayMs;

            public WalletSaveRequest(Wallet wallet, long requestedDelayMs) {
                this.requestedDelayMs = requestedDelayMs;
                this.wallet = wallet;
            }

            @Override
            public long getDelay(TimeUnit timeUnit) {
                long delayRemainingMs = this.requestedDelayMs - (System.currentTimeMillis() - this.startTimeMs);
                return timeUnit.convert(delayRemainingMs, TimeUnit.MILLISECONDS);
            }

            @Override
            public int compareTo(Delayed delayed) {
                if (delayed == this) {
                    return 0;
                }
                long delta = this.getDelay(TimeUnit.MILLISECONDS) - delayed.getDelay(TimeUnit.MILLISECONDS);
                return delta > 0L ? 1 : (delta < 0L ? -1 : 0);
            }

            public boolean equals(Object obj) {
                if (!(obj instanceof WalletSaveRequest)) {
                    return false;
                }
                WalletSaveRequest w = (WalletSaveRequest)obj;
                return w.startTimeMs == this.startTimeMs && w.requestedDelayMs == this.requestedDelayMs && w.wallet == this.wallet;
            }

            public int hashCode() {
                return Objects.hashCode((Object[])new Object[]{this.wallet, this.startTimeMs, this.requestedDelayMs});
            }
        }
    }

    public static class DefaultCoinSelector
    implements CoinSelector {
        @Override
        public CoinSelection select(BigInteger biTarget, LinkedList<TransactionOutput> candidates) {
            long target = biTarget.longValue();
            long total = 0L;
            LinkedList selected = Lists.newLinkedList();
            ArrayList<TransactionOutput> sortedOutputs = new ArrayList<TransactionOutput>(candidates);
            Collections.sort(sortedOutputs, new Comparator<TransactionOutput>(){

                @Override
                public int compare(TransactionOutput a, TransactionOutput b) {
                    int depth1 = 0;
                    int depth2 = 0;
                    TransactionConfidence conf1 = a.parentTransaction.getConfidence();
                    TransactionConfidence conf2 = b.parentTransaction.getConfidence();
                    if (conf1.getConfidenceType() == TransactionConfidence.ConfidenceType.BUILDING) {
                        depth1 = conf1.getDepthInBlocks();
                    }
                    if (conf2.getConfidenceType() == TransactionConfidence.ConfidenceType.BUILDING) {
                        depth2 = conf2.getDepthInBlocks();
                    }
                    if (depth1 < depth2) {
                        return 1;
                    }
                    if (depth1 > depth2) {
                        return -1;
                    }
                    BigInteger aHash = a.parentTransaction.getHash().toBigInteger();
                    BigInteger bHash = b.parentTransaction.getHash().toBigInteger();
                    return aHash.compareTo(bHash);
                }
            });
            for (TransactionOutput output : sortedOutputs) {
                if (total >= target) break;
                if (!this.shouldSelect(output.parentTransaction)) continue;
                selected.add(output);
                total += output.getValue().longValue();
            }
            return new CoinSelection(BigInteger.valueOf(total), selected);
        }

        protected boolean shouldSelect(Transaction tx) {
            return DefaultCoinSelector.isSelectable(tx);
        }

        public static boolean isSelectable(Transaction tx) {
            TransactionConfidence confidence = tx.getConfidence();
            TransactionConfidence.ConfidenceType type = confidence.getConfidenceType();
            boolean pending = type.equals((Object)TransactionConfidence.ConfidenceType.NOT_SEEN_IN_CHAIN) || type.equals((Object)TransactionConfidence.ConfidenceType.NOT_IN_BEST_CHAIN);
            boolean confirmed = type.equals((Object)TransactionConfidence.ConfidenceType.BUILDING);
            if (!confirmed) {
                if (!pending) {
                    return false;
                }
                if (!confidence.getSource().equals((Object)TransactionConfidence.Source.SELF)) {
                    return false;
                }
                if (confidence.numBroadcastPeers() <= 1) {
                    return false;
                }
            }
            return true;
        }
    }

    public static interface CoinSelector {
        public CoinSelection select(BigInteger var1, LinkedList<TransactionOutput> var2);
    }

    public static class CoinSelection {
        public BigInteger valueGathered;
        public List<TransactionOutput> gathered;

        public CoinSelection(BigInteger valueGathered, List<TransactionOutput> gathered) {
            this.valueGathered = valueGathered;
            this.gathered = gathered;
        }
    }
}

